const mysql = require('mysql2/promise');

// Database configuration - update these values
const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '', // Update with your password
  database: 'cement' // Update with your database name
};

async function addCustomerIdToPayments() {
  let connection;
  
  try {
    console.log('Connecting to database...');
    connection = await mysql.createConnection(dbConfig);
    console.log('Connected successfully!');

    // Check if customer_id column already exists
    console.log('\n=== Checking current table structure ===');
    const [columns] = await connection.execute('DESCRIBE payment_entries');
    const hasCustomerId = columns.some(col => col.Field === 'customer_id');
    
    if (hasCustomerId) {
      console.log('✅ customer_id column already exists in payment_entries table');
    } else {
      console.log('❌ customer_id column does not exist, adding it...');
      
      // Add customer_id column
      await connection.execute(`
        ALTER TABLE payment_entries 
        ADD COLUMN customer_id INT NULL,
        ADD CONSTRAINT fk_payment_entries_customer 
        FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL
      `);
      
      console.log('✅ Added customer_id column to payment_entries table');
      
      // Now try to populate customer_id based on existing relationships
      console.log('\n=== Populating customer_id based on existing relationships ===');
      
      // Get all payment entries
      const [payments] = await connection.execute('SELECT id, company_id, driver_name FROM payment_entries');
      console.log(`Found ${payments.length} payment entries to process`);
      
      let updatedCount = 0;
      
      for (const payment of payments) {
        if (payment.company_id) {
          // Try to find a customer that has invoices with this company
          const [customerRows] = await connection.execute(`
            SELECT DISTINCT customer_id 
            FROM invoices 
            WHERE company_id = ? 
            LIMIT 1
          `, [payment.company_id]);
          
          if (customerRows.length > 0) {
            await connection.execute(`
              UPDATE payment_entries 
              SET customer_id = ? 
              WHERE id = ?
            `, [customerRows[0].customer_id, payment.id]);
            updatedCount++;
          }
        }
      }
      
      console.log(`✅ Updated ${updatedCount} payment entries with customer_id`);
    }

    // Show final table structure
    console.log('\n=== Final table structure ===');
    const [finalColumns] = await connection.execute('DESCRIBE payment_entries');
    finalColumns.forEach(col => {
      console.log(`${col.Field}: ${col.Type} ${col.Null === 'YES' ? 'NULL' : 'NOT NULL'}`);
    });

  } catch (error) {
    console.error('❌ Error:', error.message);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed.');
    }
  }
}

// Run the script
addCustomerIdToPayments();
