const mysql = require('mysql2/promise');

async function addSampleReceipts() {
  const connection = await mysql.createConnection({
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'cement'
  });

  try {
    console.log('=== ADDING SAMPLE RECEIPTS DATA ===\n');
    
    // First, let's see what vehicles exist
    const [vehicles] = await connection.query('SELECT id, number, name FROM vehicles LIMIT 5');
    console.log('Available vehicles:');
    vehicles.forEach(vehicle => {
      console.log(`- ID: ${vehicle.id}, Number: "${vehicle.number}", Name: "${vehicle.name || 'N/A'}"`);
    });
    
    if (vehicles.length === 0) {
      console.log('No vehicles found. Please add some vehicles first.');
      return;
    }
    
    // Add sample receipts with vehicle numbers
    const sampleReceipts = [
      {
        vehicle_no: vehicles[0].number,
        bill_no: 'R001',
        payment_date: '2024-01-15',
        payment_amount: 5000,
        payment_mode: 'Cash',
        notes: 'Customer payment for delivery',
        customer_id: null,
        driver_id: null
      },
      {
        vehicle_no: vehicles[0].number,
        bill_no: 'R002',
        payment_date: '2024-01-16',
        payment_amount: 3500,
        payment_mode: 'Bank Transfer',
        notes: 'Customer payment for delivery',
        customer_id: null,
        driver_id: null
      },
      {
        vehicle_no: vehicles.length > 1 ? vehicles[1].number : vehicles[0].number,
        bill_no: 'R003',
        payment_date: '2024-01-17',
        payment_amount: 4200,
        payment_mode: 'Cash',
        notes: 'Customer payment for delivery',
        customer_id: null,
        driver_id: null
      }
    ];
    
    console.log('\nAdding sample receipts...');
    
    for (const receipt of sampleReceipts) {
      try {
        await connection.execute(`
          INSERT INTO receipts (vehicle_no, bill_no, payment_date, payment_amount, payment_mode, notes, customer_id, driver_id, created_at)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
        `, [
          receipt.vehicle_no,
          receipt.bill_no,
          receipt.payment_date,
          receipt.payment_amount,
          receipt.payment_mode,
          receipt.notes,
          receipt.customer_id,
          receipt.driver_id
        ]);
        console.log(`✓ Added receipt ${receipt.bill_no} for vehicle ${receipt.vehicle_no}`);
      } catch (error) {
        console.error(`✗ Failed to add receipt ${receipt.bill_no}:`, error.message);
      }
    }
    
    // Verify the receipts were added
    const [addedReceipts] = await connection.query(`
      SELECT id, vehicle_no, bill_no, payment_date, payment_amount, payment_mode, notes
      FROM receipts 
      WHERE vehicle_no IS NOT NULL AND vehicle_no != ""
      ORDER BY id DESC 
      LIMIT 10
    `);
    
    console.log('\n=== VERIFICATION: RECEIPTS WITH VEHICLE NUMBERS ===');
    if (addedReceipts.length > 0) {
      addedReceipts.forEach(receipt => {
        console.log(`Receipt ID: ${receipt.id}, Vehicle: "${receipt.vehicle_no}", Bill: ${receipt.bill_no}, Amount: ${receipt.payment_amount}`);
      });
    } else {
      console.log('No receipts with vehicle numbers found');
    }
    
  } catch (error) {
    console.error('Error:', error.message);
  } finally {
    await connection.end();
  }
}

addSampleReceipts();










