const mysql = require('mysql2/promise');

async function debugDriverReceipts() {
  let connection;
  
  try {
    connection = await mysql.createConnection({
      host: 'localhost',
      user: 'root',
      password: '',
      database: 'cement_management'
    });

    console.log('🔍 Debugging driver receipts...');

    // Check all drivers
    const [drivers] = await connection.execute('SELECT id, name FROM drivers ORDER BY id');
    console.log('\n📋 All drivers:');
    drivers.forEach(d => console.log(`  - ID: ${d.id}, Name: "${d.name}"`));

    // Check all receipts with driver_id
    const [receipts] = await connection.execute(`
      SELECT id, driver_id, company_id, bill_no, payment_amount, payment_date, created_at
      FROM receipts 
      WHERE driver_id IS NOT NULL
      ORDER BY created_at DESC 
      LIMIT 20
    `);
    console.log('\n📋 Receipts with driver_id:');
    receipts.forEach(r => console.log(`  - Receipt ${r.id}: Driver ID: ${r.driver_id}, Bill: ${r.bill_no}, Amount: ₹${r.payment_amount}`));

    // Check receipts without driver_id
    const [receiptsWithoutDriver] = await connection.execute(`
      SELECT id, driver_id, company_id, bill_no, payment_amount, payment_date, created_at
      FROM receipts 
      WHERE driver_id IS NULL
      ORDER BY created_at DESC 
      LIMIT 10
    `);
    console.log('\n📋 Receipts without driver_id:');
    receiptsWithoutDriver.forEach(r => console.log(`  - Receipt ${r.id}: Driver ID: ${r.driver_id}, Bill: ${r.bill_no}, Amount: ₹${r.payment_amount}`));

    // Test the exact query used in the backend route for each driver
    for (const driver of drivers) {
      console.log(`\n🔍 Testing receipts for driver: "${driver.name}" (ID: ${driver.id})`);
      
      const [driverReceipts] = await connection.execute(`
        SELECT DISTINCT
          r.id,
          r.company_id,
          r.driver_id,
          r.bill_no,
          DATE_FORMAT(r.payment_date, '%Y-%m-%d') as payment_date_formatted,
          r.payment_amount,
          r.payment_mode,
          r.reference_number,
          r.notes,
          r.bank_id,
          r.created_at,
          c.name as company_name,
          b.bank_name,
          b.account_number,
          b.ifsc_code,
          b.branch,
          b.account_name as bank_account_name,
          'receipt' as transaction_type
        FROM receipts r
        LEFT JOIN companies c ON r.company_id = c.id
        LEFT JOIN banks b ON r.bank_id = b.id
        WHERE r.driver_id = ?
        ORDER BY r.payment_date DESC, r.created_at DESC
      `, [driver.id]);
      
      console.log(`  📊 Receipts found: ${driverReceipts.length}`);
      driverReceipts.forEach(r => console.log(`    - Receipt ${r.id}: ${r.bill_no}, Amount: ₹${r.payment_amount}, Date: ${r.payment_date_formatted}`));
    }

    // Check if there are any recent receipts that might have been created
    const [recentReceipts] = await connection.execute(`
      SELECT id, driver_id, bill_no, payment_amount, payment_date, created_at
      FROM receipts 
      ORDER BY created_at DESC 
      LIMIT 10
    `);
    console.log('\n📋 Most recent receipts:');
    recentReceipts.forEach(r => console.log(`  - Receipt ${r.id}: Driver ID: ${r.driver_id}, Bill: ${r.bill_no}, Amount: ₹${r.payment_amount}, Created: ${r.created_at}`));

  } catch (error) {
    console.error('❌ Error debugging driver receipts:', error);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed');
    }
  }
}

debugDriverReceipts();
