const mysql = require('mysql2/promise');

async function fixDatabaseSchema() {
  let connection;
  
  try {
    connection = await mysql.createConnection({
      host: 'localhost',
      user: 'root',
      password: '',
      database: 'cement_management'
    });

    console.log('🔧 Fixing database schema...');

    // Check current diesel_purchases structure
    const [structure] = await connection.execute('DESCRIBE diesel_purchases');
    console.log('\n📋 Current diesel_purchases structure:');
    structure.forEach(col => console.log(`  - ${col.Field}: ${col.Type} ${col.Null === 'YES' ? 'NULL' : 'NOT NULL'}`));

    // Check if driver_name column exists
    const [columns] = await connection.execute(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'diesel_purchases'
      AND COLUMN_NAME = 'driver_name'
    `);

    if (columns.length === 0) {
      console.log('\n❌ driver_name column missing. Adding it...');
      
      try {
        await connection.execute(`
          ALTER TABLE diesel_purchases 
          ADD COLUMN driver_name VARCHAR(255) NULL 
          AFTER vehicle_id
        `);
        console.log('✅ Added driver_name column to diesel_purchases table');
      } catch (error) {
        console.log('❌ Error adding column:', error.message);
        throw error;
      }
    } else {
      console.log('\n✅ driver_name column already exists');
    }

    // Verify the column was added
    const [newStructure] = await connection.execute('DESCRIBE diesel_purchases');
    console.log('\n📋 Updated diesel_purchases structure:');
    newStructure.forEach(col => console.log(`  - ${col.Field}: ${col.Type} ${col.Null === 'YES' ? 'NULL' : 'NOT NULL'}`));

    // Check if there are any existing diesel purchases
    const [existingPurchases] = await connection.execute(`
      SELECT COUNT(*) as count FROM diesel_purchases
    `);
    console.log(`\n📊 Existing diesel purchases: ${existingPurchases[0].count}`);

    // Check if there are any diesel purchases with driver_name
    const [purchasesWithDriver] = await connection.execute(`
      SELECT COUNT(*) as count FROM diesel_purchases WHERE driver_name IS NOT NULL
    `);
    console.log(`📊 Diesel purchases with driver_name: ${purchasesWithDriver[0].count}`);

    // Show sample data if any exists
    if (existingPurchases[0].count > 0) {
      const [sampleData] = await connection.execute(`
        SELECT id, driver_name, bill_no, total_amount, bill_date 
        FROM diesel_purchases 
        ORDER BY created_at DESC 
        LIMIT 5
      `);
      console.log('\n📋 Sample diesel purchases:');
      sampleData.forEach(p => console.log(`  - ID: ${p.id}, Driver: ${p.driver_name || 'NULL'}, Bill: ${p.bill_no}, Amount: ₹${p.total_amount}`));
    }

    console.log('\n✅ Database schema fix completed!');

  } catch (error) {
    console.error('❌ Error fixing database schema:', error);
  } finally {
    if (connection) {
      await connection.end();
      console.log('Database connection closed');
    }
  }
}

fixDatabaseSchema();
