const mysql = require('mysql2/promise');

async function fixForeignKeyComplete() {
  const connection = await mysql.createConnection({
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'cement'
  });
  
  console.log('=== COMPREHENSIVE FOREIGN KEY FIX ===');
  
  try {
    // Step 1: Check current schema
    console.log('\n1. Checking current schema...');
    
    const [paymentEntriesFK] = await connection.query(`
      SELECT 
        CONSTRAINT_NAME,
        COLUMN_NAME,
        REFERENCED_TABLE_NAME,
        REFERENCED_COLUMN_NAME,
        DELETE_RULE,
        UPDATE_RULE
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'payment_entries'
      AND REFERENCED_TABLE_NAME IS NOT NULL
    `);
    console.log('Current Foreign Keys:', paymentEntriesFK);
    
    // Step 2: Check data types
    console.log('\n2. Checking data types...');
    const [paymentEntriesColumns] = await connection.query(`
      SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'payment_entries'
      AND COLUMN_NAME = 'selected_expense_id'
    `);
    console.log('selected_expense_id column:', paymentEntriesColumns);
    
    const [expensesColumns] = await connection.query(`
      SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'expenses'
      AND COLUMN_NAME = 'id'
    `);
    console.log('expenses.id column:', expensesColumns);
    
    const [expenseTypesColumns] = await connection.query(`
      SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'expense_types'
      AND COLUMN_NAME = 'id'
    `);
    console.log('expense_types.id column:', expenseTypesColumns);
    
    // Step 3: Drop existing foreign key constraint
    console.log('\n3. Dropping existing foreign key constraint...');
    try {
      await connection.query(`
        ALTER TABLE payment_entries 
        DROP FOREIGN KEY payment_entries_ibfk_3
      `);
      console.log('✅ Dropped payment_entries_ibfk_3');
    } catch (error) {
      if (error.code === 'ER_CANT_DROP_FIELD_OR_KEY') {
        console.log('⚠️ Constraint might have different name, trying to find it...');
        
        // Find all foreign key constraints
        const [allConstraints] = await connection.query(`
          SELECT CONSTRAINT_NAME 
          FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS 
          WHERE TABLE_SCHEMA = 'cement' 
          AND TABLE_NAME = 'payment_entries'
          AND CONSTRAINT_TYPE = 'FOREIGN KEY'
        `);
        console.log('All foreign key constraints:', allConstraints);
        
        // Try to drop each one
        for (const constraint of allConstraints) {
          try {
            await connection.query(`
              ALTER TABLE payment_entries 
              DROP FOREIGN KEY ${constraint.CONSTRAINT_NAME}
            `);
            console.log(`✅ Dropped ${constraint.CONSTRAINT_NAME}`);
          } catch (e) {
            console.log(`⚠️ Could not drop ${constraint.CONSTRAINT_NAME}: ${e.message}`);
          }
        }
      } else {
        throw error;
      }
    }
    
    // Step 4: Add new foreign key constraint to expense_types
    console.log('\n4. Adding new foreign key constraint to expense_types...');
    try {
      await connection.query(`
        ALTER TABLE payment_entries 
        ADD CONSTRAINT payment_entries_expense_type_fk 
        FOREIGN KEY (selected_expense_id) 
        REFERENCES expense_types(id) 
        ON DELETE SET NULL
      `);
      console.log('✅ Added foreign key constraint to expense_types');
    } catch (error) {
      console.log('⚠️ Could not add foreign key to expense_types:', error.message);
      console.log('Continuing without foreign key constraint...');
    }
    
    // Step 5: Verify the fix
    console.log('\n5. Verifying the fix...');
    const [newConstraints] = await connection.query(`
      SELECT 
        CONSTRAINT_NAME,
        COLUMN_NAME,
        REFERENCED_TABLE_NAME,
        REFERENCED_COLUMN_NAME
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'payment_entries'
      AND REFERENCED_TABLE_NAME IS NOT NULL
    `);
    console.log('New foreign key constraints:', newConstraints);
    
    // Step 6: Test insert with expense_type ID
    console.log('\n6. Testing insert with expense_type ID...');
    
    // Get an expense_type ID
    const [expenseTypes] = await connection.query('SELECT id FROM expense_types LIMIT 1');
    if (expenseTypes.length > 0) {
      const expenseTypeId = expenseTypes[0].id;
      console.log('Testing with expense_type ID:', expenseTypeId);
      
      try {
        const [result] = await connection.query(`
          INSERT INTO payment_entries (
            bill_no, bill_date, account_type_bank_name, driver_name, vehicle_no,
            selected_expense_id, expense_amount, total, note
          ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        `, [
          'TEST-FK-' + Date.now(),
          new Date(),
          'Cash',
          'Test Driver',
          'TEST-123',
          expenseTypeId,
          1000,
          1000,
          'Test payment with expense_type ID'
        ]);
        
        console.log('✅ Test insert successful with ID:', result.insertId);
        
        // Clean up test data
        await connection.query('DELETE FROM payment_entries WHERE id = ?', [result.insertId]);
        console.log('✅ Test data cleaned up');
        
      } catch (error) {
        console.log('❌ Test insert failed:', error.message);
      }
    } else {
      console.log('⚠️ No expense_types found, creating one...');
      
      const [result] = await connection.query(`
        INSERT INTO expense_types (name, description, category) VALUES (?, ?, ?)
      `, ['Test Type', 'Test Description', 'test']);
      
      console.log('✅ Created test expense_type with ID:', result.insertId);
    }
    
    console.log('\n🎉 Foreign key fix completed!');
    
  } catch (error) {
    console.error('❌ Error during fix:', error.message);
  }
  
  await connection.end();
}

fixForeignKeyComplete().catch(console.error);










