const mysql = require('mysql2/promise');

async function fixNullExpenseIds() {
  const connection = await mysql.createConnection({
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'cement'
  });
  
  console.log('=== FIXING NULL EXPENSE IDS ===');
  
  try {
    // Step 1: Remove the foreign key constraint
    console.log('\n1. Removing foreign key constraint...');
    try {
      await connection.query(`
        ALTER TABLE payment_entries 
        DROP FOREIGN KEY payment_entries_ibfk_3
      `);
      console.log('✅ Dropped payment_entries_ibfk_3');
    } catch (error) {
      if (error.code === 'ER_CANT_DROP_FIELD_OR_KEY') {
        console.log('⚠️ Constraint not found or already removed');
      } else {
        console.log('⚠️ Error dropping constraint:', error.message);
      }
    }
    
    // Step 2: Get all expense types
    console.log('\n2. Getting expense types...');
    const [expenseTypes] = await connection.query('SELECT * FROM expense_types');
    console.log('Available expense types:', expenseTypes);
    
    // Step 3: Fix existing payment entries with notes that contain expense type names
    console.log('\n3. Fixing existing payment entries...');
    
    for (const expenseType of expenseTypes) {
      const expenseTypeName = expenseType.name;
      console.log(`\nProcessing expense type: ${expenseTypeName}`);
      
      // Find payment entries with notes containing this expense type name
      const [paymentEntries] = await connection.query(`
        SELECT id, note, selected_expense_id 
        FROM payment_entries 
        WHERE note LIKE ? 
        AND selected_expense_id IS NULL
      `, [`%${expenseTypeName}%`]);
      
      console.log(`Found ${paymentEntries.length} payment entries for ${expenseTypeName}`);
      
      for (const payment of paymentEntries) {
        console.log(`Updating payment ID ${payment.id} with expense_type_id ${expenseType.id}`);
        
        // Update the payment entry with the expense type ID
        await connection.query(`
          UPDATE payment_entries 
          SET selected_expense_id = ? 
          WHERE id = ?
        `, [expenseType.id, payment.id]);
        
        console.log(`✅ Updated payment ID ${payment.id}`);
      }
    }
    
    // Step 4: Verify the fixes
    console.log('\n4. Verifying fixes...');
    const [updatedPayments] = await connection.query(`
      SELECT 
        id,
        bill_no,
        selected_expense_id,
        note,
        expense_amount
      FROM payment_entries 
      WHERE selected_expense_id IS NOT NULL
      ORDER BY created_at DESC
    `);
    
    console.log(`\nFound ${updatedPayments.length} payment entries with expense IDs:`);
    updatedPayments.forEach((payment, index) => {
      console.log(`${index + 1}. Payment ID: ${payment.id}, Bill: ${payment.bill_no}, Expense ID: ${payment.selected_expense_id}, Note: ${payment.note}`);
    });
    
    // Step 5: Check remaining NULL entries
    const [nullPayments] = await connection.query(`
      SELECT 
        id,
        bill_no,
        note,
        expense_amount
      FROM payment_entries 
      WHERE selected_expense_id IS NULL
      AND note IS NOT NULL
      AND note != ''
      ORDER BY created_at DESC
    `);
    
    console.log(`\nRemaining ${nullPayments.length} payment entries with NULL expense IDs but with notes:`);
    nullPayments.forEach((payment, index) => {
      console.log(`${index + 1}. Payment ID: ${payment.id}, Bill: ${payment.bill_no}, Note: ${payment.note}`);
    });
    
    console.log('\n🎉 Fix completed!');
    
  } catch (error) {
    console.error('❌ Error:', error.message);
  }
  
  await connection.end();
}

fixNullExpenseIds().catch(console.error);










