const mysql = require('mysql2/promise');

async function fixPaymentExpenseForeignKey() {
  const connection = await mysql.createConnection({
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'cement'
  });
  
  console.log('=== FIXING PAYMENT ENTRIES FOREIGN KEY ===');
  
  try {
    // First, check current foreign key constraints
    console.log('Checking current foreign key constraints...');
    const [constraints] = await connection.query(`
      SELECT 
        CONSTRAINT_NAME,
        COLUMN_NAME,
        REFERENCED_TABLE_NAME,
        REFERENCED_COLUMN_NAME
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'payment_entries'
      AND CONSTRAINT_NAME LIKE '%expense%'
    `);
    
    console.log('Current expense-related foreign keys:', constraints);
    
    // Drop the existing foreign key constraint
    if (constraints.length > 0) {
      console.log('Dropping existing foreign key constraint...');
      await connection.query(`
        ALTER TABLE payment_entries 
        DROP FOREIGN KEY payment_entries_ibfk_3
      `);
      console.log('✅ Dropped existing foreign key constraint');
    }
    
    // Add new foreign key constraint pointing to expense_types
    console.log('Adding new foreign key constraint to expense_types...');
    await connection.query(`
      ALTER TABLE payment_entries 
      ADD CONSTRAINT payment_entries_expense_type_fk 
      FOREIGN KEY (selected_expense_id) 
      REFERENCES expense_types(id) 
      ON DELETE SET NULL
    `);
    console.log('✅ Added new foreign key constraint to expense_types');
    
    // Verify the new constraint
    const [newConstraints] = await connection.query(`
      SELECT 
        CONSTRAINT_NAME,
        COLUMN_NAME,
        REFERENCED_TABLE_NAME,
        REFERENCED_COLUMN_NAME
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
      WHERE TABLE_SCHEMA = 'cement' 
      AND TABLE_NAME = 'payment_entries'
      AND CONSTRAINT_NAME LIKE '%expense%'
    `);
    
    console.log('New expense-related foreign keys:', newConstraints);
    
  } catch (error) {
    console.error('Error:', error.message);
    
    // If the constraint name is different, try to find and drop it
    if (error.code === 'ER_CANT_DROP_FIELD_OR_KEY') {
      console.log('Trying to find the correct constraint name...');
      const [allConstraints] = await connection.query(`
        SELECT CONSTRAINT_NAME 
        FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS 
        WHERE TABLE_SCHEMA = 'cement' 
        AND TABLE_NAME = 'payment_entries'
        AND CONSTRAINT_TYPE = 'FOREIGN KEY'
      `);
      
      console.log('All foreign key constraints:', allConstraints);
    }
  }
  
  await connection.end();
}

fixPaymentExpenseForeignKey().catch(console.error);

