const mysql = require('mysql2/promise');

async function migratePaymentEntries() {
  let connection;
  
  try {
    connection = await mysql.createConnection({
      host: 'localhost',
      user: 'root',
      password: '',
      database: 'cement_management'
    });

    console.log('🔄 Starting payment_entries table migration...\n');

    // Check if new columns already exist
    const [columns] = await connection.execute(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'payment_entries'
    `);
    
    const existingColumns = columns.map(col => col.COLUMN_NAME);
    console.log('📋 Existing columns:', existingColumns);

    // Add new columns if they don't exist
    const newColumns = [
      { name: 'selected_expense_id', type: 'INT', nullable: true },
      { name: 'expense_amount', type: 'DECIMAL(10,2)', nullable: true, default: '0' }
    ];

    for (const column of newColumns) {
      if (!existingColumns.includes(column.name)) {
        const alterQuery = `ALTER TABLE payment_entries ADD COLUMN ${column.name} ${column.type}${column.nullable ? ' NULL' : ' NOT NULL'}${column.default ? ` DEFAULT ${column.default}` : ''}`;
        
        console.log(`➕ Adding column: ${column.name}`);
        await connection.execute(alterQuery);
        console.log(`✅ Added column: ${column.name}`);
      } else {
        console.log(`⏭️  Column ${column.name} already exists`);
      }
    }

    // Add foreign key constraint for selected_expense_id if it doesn't exist
    const [constraints] = await connection.execute(`
      SELECT CONSTRAINT_NAME 
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'payment_entries' 
      AND COLUMN_NAME = 'selected_expense_id'
      AND REFERENCED_TABLE_NAME IS NOT NULL
    `);

    if (constraints.length === 0) {
      try {
        console.log('🔗 Adding foreign key constraint for selected_expense_id...');
        await connection.execute(`
          ALTER TABLE payment_entries 
          ADD CONSTRAINT fk_payment_expense 
          FOREIGN KEY (selected_expense_id) REFERENCES expenses(id) ON DELETE SET NULL
        `);
        console.log('✅ Foreign key constraint added');
      } catch (error) {
        console.log('⚠️  Could not add foreign key constraint (expenses table might not exist):', error.message);
      }
    } else {
      console.log('⏭️  Foreign key constraint already exists');
    }

    console.log('\n🎉 Migration completed successfully!');
    
  } catch (error) {
    console.error('❌ Migration failed:', error);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

// Run migration if called directly
if (require.main === module) {
  migratePaymentEntries();
}

module.exports = { migratePaymentEntries };
