const mysql = require('mysql2/promise');

// Database configuration - update these values
const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '', // Update with your password
  database: 'cement' // Update with your database name
};

async function populateCustomerIds() {
  let connection;
  
  try {
    console.log('Connecting to database...');
    connection = await mysql.createConnection(dbConfig);
    console.log('Connected successfully!');

    console.log('\n=== Populating customer_id in payment_entries table ===');
    
    // Get all payment entries that don't have customer_id set
    const [payments] = await connection.execute(`
      SELECT pe.id, pe.company_id, pe.bill_no
      FROM payment_entries pe
      WHERE pe.customer_id IS NULL
    `);
    
    console.log(`Found ${payments.length} payment entries without customer_id`);
    
    let updatedCount = 0;
    
    for (const payment of payments) {
      if (payment.company_id) {
        // Try to find a customer that has invoices with this company
        const [customerRows] = await connection.execute(`
          SELECT DISTINCT customer_id, c.name as customer_name
          FROM invoices i
          JOIN customers c ON i.customer_id = c.id
          WHERE i.company_id = ? 
          LIMIT 1
        `, [payment.company_id]);
        
        if (customerRows.length > 0) {
          await connection.execute(`
            UPDATE payment_entries 
            SET customer_id = ? 
            WHERE id = ?
          `, [customerRows[0].customer_id, payment.id]);
          
          console.log(`✅ Updated Payment ID ${payment.id} (Bill: ${payment.bill_no}) with Customer: ${customerRows[0].customer_name} (ID: ${customerRows[0].customer_id})`);
          updatedCount++;
        } else {
          console.log(`⚠️  No customer found for Payment ID ${payment.id} (Bill: ${payment.bill_no}) with Company ID: ${payment.company_id}`);
        }
      } else {
        console.log(`⚠️  Payment ID ${payment.id} (Bill: ${payment.bill_no}) has no company_id`);
      }
    }
    
    console.log(`\n✅ Successfully updated ${updatedCount} payment entries with customer_id`);
    
    // Show final status
    console.log('\n=== Final Status ===');
    const [finalCount] = await connection.execute(`
      SELECT 
        COUNT(*) as total,
        COUNT(CASE WHEN customer_id IS NOT NULL THEN 1 END) as with_customer,
        COUNT(CASE WHEN customer_id IS NULL THEN 1 END) as without_customer
      FROM payment_entries
    `);
    
    console.log(`Total payment entries: ${finalCount[0].total}`);
    console.log(`With customer_id: ${finalCount[0].with_customer}`);
    console.log(`Without customer_id: ${finalCount[0].without_customer}`);

  } catch (error) {
    console.error('❌ Error:', error.message);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed.');
    }
  }
}

// Run the script
populateCustomerIds();
