const express = require('express');
const router = express.Router();
const mysql = require('mysql2/promise');

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// Get all customers
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT * FROM customers ORDER BY created_at DESC');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching customers:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get single customer by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  
  try {
    const [rows] = await db.query('SELECT * FROM customers WHERE id = ?', [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Customer not found' });
    }
    
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching customer:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add new customer
router.post('/', async (req, res) => {
  const { name, contact, email, location } = req.body;
  
  if (!name || !contact || !location) {
    return res.status(400).json({ message: 'Customer name, contact, and location are required' });
  }

  try {
    const [result] = await db.query(
      'INSERT INTO customers (name, contact, email, location) VALUES (?, ?, ?, ?)',
      [name, contact, email || null, location]
    );
    
    res.status(201).json({ 
      message: 'Customer added successfully',
      id: result.insertId,
      customer: { id: result.insertId, name, contact, email, location }
    });
  } catch (err) {
    console.error('Error adding customer:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update customer
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const { name, contact, email, location } = req.body;
  
  if (!name || !contact || !location) {
    return res.status(400).json({ message: 'Customer name, contact, and location are required' });
  }

  try {
    // Check if customer exists
    const [existing] = await db.query('SELECT id FROM customers WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Customer not found' });
    }

    await db.query(
      'UPDATE customers SET name = ?, contact = ?, email = ?, location = ? WHERE id = ?',
      [name, contact, email || null, location, id]
    );
    
    res.json({ message: 'Customer updated successfully' });
  } catch (err) {
    console.error('Error updating customer:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete customer
router.delete('/:id', async (req, res) => {
  const { id } = req.params;

  try {
    const [result] = await db.query('DELETE FROM customers WHERE id = ?', [id]);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Customer not found' });
    }
    
    res.json({ message: 'Customer deleted successfully' });
  } catch (err) {
    console.error('Error deleting customer:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 