const express = require('express');
const router = express.Router();
let db = null;

// Function to set database connection
const setDb = (database) => {
  db = database;
  // Initialize table and default data when database is set
  initializeExpenseTypes();
};

// Function to initialize expense types table and default data
const initializeExpenseTypes = async () => {
  if (!db) return;
  
  try {
    console.log('🔧 Initializing expense_types table...');
    
    // First, check if table exists
    const [tables] = await db.execute("SHOW TABLES LIKE 'expense_types'");
    
    if (tables.length === 0) {
      console.log('📋 Creating expense_types table...');
      
      // Create table with all necessary columns
      await db.execute(`
        CREATE TABLE expense_types (
          id INT AUTO_INCREMENT PRIMARY KEY,
          name VARCHAR(255) NOT NULL UNIQUE,
          description TEXT,
          category VARCHAR(100) DEFAULT 'general',
          company_id INT DEFAULT NULL,
          is_active BOOLEAN DEFAULT TRUE,
          sort_order INT DEFAULT 0,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
      `);
      
      console.log('✅ expense_types table created successfully');
    } else {
      console.log('✅ expense_types table already exists');
      
      // Check and add missing columns if table exists but is missing columns
      await addMissingColumns();
    }
    
    // Insert default expense types only if table is empty
    const [countResult] = await db.execute('SELECT COUNT(*) as count FROM expense_types');
    const currentCount = countResult[0].count;
    
    if (currentCount === 0) {
      console.log('📝 Inserting default expense types...');
      
      const defaultExpenses = [
        { name: 'Driver Advance', description: 'Advance payment to drivers', category: 'driver' },
        { name: 'Unloading Charges', description: 'Charges for unloading goods', category: 'operational' },
        { name: 'Diesel', description: 'Fuel expenses', category: 'fuel' },
        { name: 'Toll Gate', description: 'Toll gate charges', category: 'operational' },
        { name: 'Tyres Purchase', description: 'Vehicle tyre expenses', category: 'vehicle' },
        { name: 'Wheel Alignment', description: 'Vehicle maintenance - wheel alignment', category: 'vehicle' },
        { name: 'Servicing Charges', description: 'Vehicle servicing and maintenance', category: 'vehicle' },
        { name: 'Other Expenses', description: 'Miscellaneous expenses', category: 'general' },
        { name: 'Road Tax', description: 'Vehicle road tax', category: 'vehicle' },
        { name: 'Insurance', description: 'Vehicle insurance', category: 'vehicle' }
      ];
      
      for (const expense of defaultExpenses) {
        try {
          await db.execute(
            'INSERT INTO expense_types (name, description, category, is_active, sort_order) VALUES (?, ?, ?, ?, ?)',
            [expense.name, expense.description, expense.category, true, defaultExpenses.indexOf(expense) + 1]
          );
          
          // Add corresponding column to payment_entries table
          await addColumnToPaymentEntries(expense.name);
          
          console.log(`✅ ${expense.name}`);
        } catch (error) {
          console.error(`❌ Error inserting ${expense.name}:`, error.message);
        }
      }
    } else {
      console.log(`⚠️  Table already has ${currentCount} expense types, skipping default insertion`);
    }
    
    // Verify the setup
    const [rows] = await db.execute('SELECT COUNT(*) as count FROM expense_types');
    console.log(`📊 Total expense types in database: ${rows[0].count}`);
    console.log('✅ Expense types initialization complete!');
    
  } catch (error) {
    console.error('❌ Error initializing expense types:', error);
    throw error; // Re-throw to handle in calling functions
  }
};

// Function to add missing columns to existing table
const addMissingColumns = async () => {
  try {
    console.log('🔍 Checking for missing columns...');
    
    // Get current table structure
    const [columns] = await db.execute("DESCRIBE expense_types");
    const existingColumns = columns.map(col => col.Field);
    
    const requiredColumns = [
      { name: 'company_id', definition: 'ADD COLUMN company_id INT DEFAULT NULL' },
      { name: 'is_active', definition: 'ADD COLUMN is_active BOOLEAN DEFAULT TRUE' },
      { name: 'sort_order', definition: 'ADD COLUMN sort_order INT DEFAULT 0' }
    ];
    
    for (const column of requiredColumns) {
      if (!existingColumns.includes(column.name)) {
        console.log(`➕ Adding missing column: ${column.name}`);
        await db.execute(`ALTER TABLE expense_types ${column.definition}`);
        console.log(`✅ Added column: ${column.name}`);
      } else {
        console.log(`✅ Column ${column.name} already exists`);
      }
    }
    
  } catch (error) {
    console.error('❌ Error adding missing columns:', error);
  }
};

// GET /expense-types - Get all expense types
router.get('/', async (req, res) => {
  if (!db) {
    return res.status(500).json({ error: 'Database not initialized' });
  }
  
  try {
    // Ensure table exists before querying
    await initializeExpenseTypes();
    
    const query = 'SELECT * FROM expense_types WHERE is_active = TRUE ORDER BY sort_order ASC, name ASC';
    const [rows] = await db.execute(query);
    res.json(rows);
  } catch (error) {
    console.error('Error fetching expense types:', error);
    res.status(500).json({ error: 'Failed to fetch expense types' });
  }
});

// Function to add column to payment_entries table
const addColumnToPaymentEntries = async (expenseName) => {
  try {
    // Convert expense name to column name (lowercase, replace spaces with underscores)
    const columnName = expenseName.toLowerCase().replace(/\s+/g, '_');
    
    console.log(`🔧 Adding column '${columnName}' to payment_entries table...`);
    
    // Check if column already exists
    const [columns] = await db.execute("DESCRIBE payment_entries");
    const existingColumns = columns.map(col => col.Field);
    
    if (existingColumns.includes(columnName)) {
      console.log(`✅ Column '${columnName}' already exists in payment_entries table`);
      return true;
    }
    
    // Add the column
    const alterQuery = `ALTER TABLE payment_entries ADD COLUMN \`${columnName}\` DECIMAL(10,2) DEFAULT 0.00`;
    await db.execute(alterQuery);
    
    console.log(`✅ Successfully added column '${columnName}' to payment_entries table`);
    return true;
  } catch (error) {
    console.error(`❌ Error adding column '${columnName}' to payment_entries table:`, error);
    return false;
  }
};

// POST /expense-types - Create a new expense type
router.post('/', async (req, res) => {
  if (!db) {
    return res.status(500).json({ error: 'Database not initialized' });
  }
  
  try {
    // Ensure table exists before creating
    await initializeExpenseTypes();
    
    const { name, description, category, company_id, is_active } = req.body;
    
    if (!name || !name.trim()) {
      return res.status(400).json({ error: 'Expense name is required' });
    }

    // Check if expense type already exists
    const checkQuery = 'SELECT id FROM expense_types WHERE name = ?';
    const [existing] = await db.execute(checkQuery, [name.trim()]);
    
    if (existing.length > 0) {
      return res.status(400).json({ error: 'Expense type already exists' });
    }

    // Get next sort order
    const [sortResult] = await db.execute('SELECT MAX(sort_order) as max_sort FROM expense_types');
    const nextSortOrder = (sortResult[0].max_sort || 0) + 1;

    const insertQuery = `
      INSERT INTO expense_types (name, description, category, company_id, is_active, sort_order, created_at) 
      VALUES (?, ?, ?, ?, ?, ?, NOW())
    `;
    
    const [result] = await db.execute(insertQuery, [
      name.trim(),
      description ? description.trim() : null,
      category ? category.trim() : 'general',
      company_id || null,
      is_active !== undefined ? is_active : true,
      nextSortOrder
    ]);

    // Add column to payment_entries table
    const columnAdded = await addColumnToPaymentEntries(name.trim());
    
    if (!columnAdded) {
      console.warn(`⚠️ Failed to add column for expense type: ${name.trim()}`);
    }

    res.status(201).json({
      id: result.insertId,
      name: name.trim(),
      description: description ? description.trim() : null,
      category: category ? category.trim() : 'general',
      company_id: company_id || null,
      is_active: is_active !== undefined ? is_active : true,
      sort_order: nextSortOrder,
      created_at: new Date(),
      column_name: name.trim().toLowerCase().replace(/\s+/g, '_')
    });
  } catch (error) {
    console.error('Error creating expense type:', error);
    res.status(500).json({ error: 'Failed to create expense type' });
  }
});

// PUT /expense-types/:id - Update an expense type
router.put('/:id', async (req, res) => {
  if (!db) {
    return res.status(500).json({ error: 'Database not initialized' });
  }
  
  try {
    const { id } = req.params;
    const { name, description, category } = req.body;
    
    if (!name || !name.trim()) {
      return res.status(400).json({ error: 'Expense name is required' });
    }

    // Check if expense type exists
    const checkQuery = 'SELECT id FROM expense_types WHERE id = ?';
    const [existing] = await db.execute(checkQuery, [id]);
    
    if (existing.length === 0) {
      return res.status(404).json({ error: 'Expense type not found' });
    }

    // Check if another expense type with same name exists
    const duplicateQuery = 'SELECT id FROM expense_types WHERE name = ? AND id != ?';
    const [duplicate] = await db.execute(duplicateQuery, [name.trim(), id]);
    
    if (duplicate.length > 0) {
      return res.status(400).json({ error: 'Expense type with this name already exists' });
    }

    const updateQuery = `
      UPDATE expense_types 
      SET name = ?, description = ?, category = ?, updated_at = NOW()
      WHERE id = ?
    `;
    
    await db.execute(updateQuery, [
      name.trim(),
      description ? description.trim() : null,
      category ? category.trim() : 'general',
      id
    ]);

    res.json({
      id: parseInt(id),
      name: name.trim(),
      description: description ? description.trim() : null,
      category: category ? category.trim() : 'general',
      updated_at: new Date()
    });
  } catch (error) {
    console.error('Error updating expense type:', error);
    res.status(500).json({ error: 'Failed to update expense type' });
  }
});

// DELETE /expense-types/:id - Delete an expense type
router.delete('/:id', async (req, res) => {
  if (!db) {
    return res.status(500).json({ error: 'Database not initialized' });
  }
  
  try {
    const { id } = req.params;

    // Check if expense type exists
    const checkQuery = 'SELECT id FROM expense_types WHERE id = ?';
    const [existing] = await db.execute(checkQuery, [id]);
    
    if (existing.length === 0) {
      return res.status(404).json({ error: 'Expense type not found' });
    }

    // Note: Removed usage check - expense types can be deleted freely

    const deleteQuery = 'DELETE FROM expense_types WHERE id = ?';
    await db.execute(deleteQuery, [id]);

    res.json({ message: 'Expense type deleted successfully' });
  } catch (error) {
    console.error('Error deleting expense type:', error);
    res.status(500).json({ error: 'Failed to delete expense type' });
  }
});

// GET /expense-types/stats - Get expense type usage statistics
router.get('/stats', async (req, res) => {
  if (!db) {
    return res.status(500).json({ error: 'Database not initialized' });
  }
  
  try {
    const statsQuery = `
      SELECT 
        'driver_advance' as expense_type,
        COUNT(*) as count,
        SUM(driver_advance) as total_amount
      FROM payment_entries WHERE driver_advance > 0
      UNION ALL
      SELECT 
        'unloading_charges' as expense_type,
        COUNT(*) as count,
        SUM(unloading_charges) as total_amount
      FROM payment_entries WHERE unloading_charges > 0
      UNION ALL
      SELECT 
        'diesel' as expense_type,
        COUNT(*) as count,
        SUM(diesel) as total_amount
      FROM payment_entries WHERE diesel > 0
      UNION ALL
      SELECT 
        'toll_gate' as expense_type,
        COUNT(*) as count,
        SUM(toll_gate) as total_amount
      FROM payment_entries WHERE toll_gate > 0
      UNION ALL
      SELECT 
        'tyres_purchase' as expense_type,
        COUNT(*) as count,
        SUM(tyres_purchase) as total_amount
      FROM payment_entries WHERE tyres_purchase > 0
      UNION ALL
      SELECT 
        'wheel_alignment' as expense_type,
        COUNT(*) as count,
        SUM(wheel_alignment) as total_amount
      FROM payment_entries WHERE wheel_alignment > 0
      UNION ALL
      SELECT 
        'servicing_charges' as expense_type,
        COUNT(*) as count,
        SUM(servicing_charges) as total_amount
      FROM payment_entries WHERE servicing_charges > 0
      UNION ALL
      SELECT 
        'other_expenses' as expense_type,
        COUNT(*) as count,
        SUM(other_expenses) as total_amount
      FROM payment_entries WHERE other_expenses > 0
      UNION ALL
      SELECT 
        'road_tax' as expense_type,
        COUNT(*) as count,
        SUM(road_tax) as total_amount
      FROM payment_entries WHERE road_tax > 0
      UNION ALL
      SELECT 
        'insurance' as expense_type,
        COUNT(*) as count,
        SUM(insurance) as total_amount
      FROM payment_entries WHERE insurance > 0
      ORDER BY total_amount DESC
    `;
    
    const [rows] = await db.execute(statsQuery);
    res.json(rows);
  } catch (error) {
    console.error('Error fetching expense stats:', error);
    res.status(500).json({ error: 'Failed to fetch expense statistics' });
  }
});

module.exports = { router, setDb };


