
const express = require('express');
const router = express.Router();
const mysql = require('mysql2/promise');
const multer = require('multer');
const path = require('path');

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// File filter for multer
const fileFilter = (req, file, cb) => {
  const allowedTypes = /jpeg|jpg|png|pdf/;
  const ext = path.extname(file.originalname).toLowerCase();
  if (allowedTypes.test(ext)) {
    cb(null, true);
  } else {
    cb(new Error('Only jpg, jpeg, png, and pdf files are allowed'));
  }
};

// Multer setup for file uploads (images and PDFs)
const storage = multer.memoryStorage();
const upload = multer({ 
  storage, 
  fileFilter,
  limits: {
    fileSize: 10 * 1024 * 1024 // 10MB limit
  }
});

// Add endpoint to get all unique destinations
router.get('/destinations', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT DISTINCT destination FROM invoices WHERE destination IS NOT NULL AND destination != "" ORDER BY destination ASC');
    res.json(rows.map(r => r.destination));
  } catch (err) {
    console.error('Error fetching destinations:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Test endpoint to verify route is working
router.get('/test', (req, res) => {
  res.json({ message: 'Invoices route is working', timestamp: new Date().toISOString() });
});

// Get all invoices with related data
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT 
        i.id,
        i.supplier_invoice_number,
        i.supplier_invoice_date,
        i.invoice_date,
        i.company_id,
        i.customer_id,
        i.from_location,
        i.destination,
        i.quantity,
        i.rate,
        i.amount,
        i.total,
        i.advance_paid,
        i.balance,
        i.driver_id,
        i.vehicle_id,
        i.driver_charges,
        i.status,
        i.created_at,
        i.bill_no,
        i.bill_created_at,
        i.settings_id,
        i.paid_date,
        c.name as customer_name,
        c.contact as customer_contact,
        c.location as customer_location,
        d.name as driver_name,
        d.phone as driver_phone,
        v.number as vehicle_number,
        v.name as vehicle_name,
        co.name as company_name
      FROM invoices i
      LEFT JOIN customers c ON i.customer_id = c.id
      LEFT JOIN drivers d ON i.driver_id = d.id
      LEFT JOIN vehicles v ON i.vehicle_id = v.id
      LEFT JOIN companies co ON i.company_id = co.id
      ORDER BY i.created_at DESC
    `);
    
    // Parse JSON fields and handle dates properly
    const invoices = rows.map(invoice => {
      // Convert created_at to Indian time
      const createdAt = new Date(invoice.created_at);
      const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
      
      // Handle supplier_invoice_date - return exactly as stored in database (no timezone conversion)
      let supplierInvoiceDate = invoice.supplier_invoice_date;
      if (supplierInvoiceDate) {
        // If it's a Date object, format it as YYYY-MM-DD without timezone conversion
        if (supplierInvoiceDate instanceof Date) {
          const year = supplierInvoiceDate.getFullYear();
          const month = String(supplierInvoiceDate.getMonth() + 1).padStart(2, '0');
          const day = String(supplierInvoiceDate.getDate()).padStart(2, '0');
          supplierInvoiceDate = `${year}-${month}-${day}`;
        } else if (typeof supplierInvoiceDate === 'string') {
          // If it's already a string, ensure it's in YYYY-MM-DD format
          supplierInvoiceDate = supplierInvoiceDate.slice(0, 10);
        }
      }
      
      // Safely parse driver_charges
      let driverCharges = [];
      try {
        driverCharges = invoice.driver_charges ? JSON.parse(invoice.driver_charges) : [];
      } catch (error) {
        console.error('Error parsing driver_charges for invoice', invoice.id, ':', error);
        driverCharges = [];
      }
      
      return {
        ...invoice,
        supplier_invoice_date: supplierInvoiceDate,
        created_at: indianCreatedAt.toISOString(),
        driver_charges: driverCharges
      };
    });
    
    res.json(invoices);
  } catch (err) {
    console.error('Error fetching invoices:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get single invoice by ID with related data
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        i.id,
        i.supplier_invoice_number,
        i.supplier_invoice_date,
        i.invoice_date,
        i.company_id,
        i.customer_id,
        i.from_location,
        i.destination,
        i.quantity,
        i.rate,
        i.amount,
        i.total,
        i.advance_paid,
        i.balance,
        i.driver_id,
        i.vehicle_id,
        i.driver_charges,
        i.status,
        i.created_at,
        i.bill_no,
        i.bill_created_at,
        i.settings_id,
        i.paid_date,
        c.name as customer_name,
        c.contact as customer_contact,
        c.location as customer_location,
        d.name as driver_name,
        d.phone as driver_phone,
        v.number as vehicle_number,
        v.name as vehicle_name,
        co.name as company_name
      FROM invoices i
      LEFT JOIN customers c ON i.customer_id = c.id
      LEFT JOIN drivers d ON i.driver_id = d.id
      LEFT JOIN vehicles v ON i.vehicle_id = v.id
      LEFT JOIN companies co ON i.company_id = co.id
      WHERE i.id = ?
    `, [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Invoice not found' });
    }
    
    const invoice = rows[0];
    
    // Convert created_at to Indian time
    const createdAt = new Date(invoice.created_at);
    const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
    
    // Handle supplier_invoice_date - return exactly as stored in database (no timezone conversion)
    let supplierInvoiceDate = invoice.supplier_invoice_date;
    if (supplierInvoiceDate) {
      // If it's a Date object, format it as YYYY-MM-DD without timezone conversion
      if (supplierInvoiceDate instanceof Date) {
        const year = supplierInvoiceDate.getFullYear();
        const month = String(supplierInvoiceDate.getMonth() + 1).padStart(2, '0');
        const day = String(supplierInvoiceDate.getDate()).padStart(2, '0');
        supplierInvoiceDate = `${year}-${month}-${day}`;
      } else if (typeof supplierInvoiceDate === 'string') {
        // If it's already a string, ensure it's in YYYY-MM-DD format
        supplierInvoiceDate = supplierInvoiceDate.slice(0, 10);
      }
    }
    
    invoice.supplier_invoice_date = supplierInvoiceDate;
    invoice.created_at = indianCreatedAt.toISOString();
    
    // Safely parse driver_charges
    try {
    invoice.driver_charges = invoice.driver_charges ? JSON.parse(invoice.driver_charges) : [];
    } catch (error) {
      console.error('Error parsing driver_charges for invoice', id, ':', error);
      invoice.driver_charges = [];
    }
    
    res.json(invoice);
  } catch (err) {
    console.error('Error fetching invoice:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Create new invoice
router.post('/', upload.fields([
  { name: 'image', maxCount: 1 },
  { name: 'pdf', maxCount: 1 }
]), async (req, res) => {
  const {
    company_id,
    customer_id,
    from_location,
    destination,
    supplier_invoice_date,
    quantity,
    rate,
    amount,
    total,
    advance_paid,
    balance,
    driver_id,
    vehicle_id,
    supplier_invoice_number,
    driver_charges = []
  } = req.body;
  
  // Parse driver_charges if it's a string (from FormData)
  let parsedDriverCharges = [];
  if (driver_charges) {
    try {
      parsedDriverCharges = typeof driver_charges === 'string' ? JSON.parse(driver_charges) : driver_charges;
      console.log('✅ Parsed driver_charges:', parsedDriverCharges);
    } catch (error) {
      console.error('❌ Error parsing driver_charges:', error);
      parsedDriverCharges = [];
    }
  }
  
  // Debug logging
  console.log('Invoice creation request body:', req.body);
  console.log('supplier_invoice_date received:', supplier_invoice_date);
  console.log('driver_id received:', driver_id, 'type:', typeof driver_id);
  console.log('vehicle_id received:', vehicle_id, 'type:', typeof vehicle_id);
  console.log('driver_charges received:', driver_charges, 'type:', typeof driver_charges);
  console.log('parsedDriverCharges:', parsedDriverCharges);
  
  let imageBuffer = null;
  let pdfBuffer = null;
  let fileType = null;
  
  // Handle image file
  if (req.files && req.files.image && req.files.image[0]) {
    imageBuffer = req.files.image[0].buffer;
    fileType = 'image';
    console.log('Image file received:', req.files.image[0].originalname);
  }
  
  // Handle PDF file
  if (req.files && req.files.pdf && req.files.pdf[0]) {
    pdfBuffer = req.files.pdf[0].buffer;
    fileType = 'pdf';
    console.log('PDF file received:', req.files.pdf[0].originalname);
  }

  // Convert empty strings to null for optional fields
  const cleanDriverId = driver_id && driver_id.trim() !== '' ? driver_id : null;
  const cleanVehicleId = vehicle_id && vehicle_id.trim() !== '' ? vehicle_id : null;
  
  console.log('cleanDriverId:', cleanDriverId);
  console.log('cleanVehicleId:', cleanVehicleId);

  // Validation
  if (!company_id || !customer_id || !from_location || !destination || !supplier_invoice_date || 
      !quantity || !rate || !supplier_invoice_number) {
    return res.status(400).json({ 
      message: 'Company, customer, from location, destination, supplier invoice date, quantity, rate, and supplier invoice number are required' 
    });
  }

  try {
    // Check if supplier invoice number already exists
    const [existingSupplierInvoice] = await db.query(
      'SELECT id FROM invoices WHERE supplier_invoice_number = ?', 
      [supplier_invoice_number]
    );
    if (existingSupplierInvoice.length > 0) {
      return res.status(400).json({ message: 'Supplier invoice number already exists' });
    }

    // Check if company exists
    console.log('🔍 Invoice - Validating company_id:', company_id, 'type:', typeof company_id);
    const [companyExists] = await db.query('SELECT id, company_name FROM company_details WHERE id = ?', [company_id]);
    console.log('📋 Invoice - Company check result:', companyExists);
    
    if (companyExists.length === 0) {
      // Get available companies for debugging
      const [availableCompanies] = await db.query('SELECT id, company_name FROM company_details ORDER BY id DESC LIMIT 10');
      console.log('📋 Available companies:', availableCompanies);
      
      return res.status(400).json({ 
        message: 'Company not found',
        company_id: company_id,
        company_id_type: typeof company_id,
        available_companies: availableCompanies
      });
    }
    console.log('✅ Invoice - Company validation passed:', companyExists[0]);

    // Check if customer exists
    const [customerExists] = await db.query('SELECT id FROM customers WHERE id = ?', [customer_id]);
    if (customerExists.length === 0) {
      return res.status(400).json({ message: 'Customer not found' });
    }

    // Check if driver exists (if provided)
    if (cleanDriverId) {
      const [driverExists] = await db.query('SELECT id FROM drivers WHERE id = ?', [cleanDriverId]);
      if (driverExists.length === 0) {
        return res.status(400).json({ message: 'Driver not found' });
      }
    }

    // Check if vehicle exists (if provided)
    if (cleanVehicleId) {
      const [vehicleExists] = await db.query('SELECT id FROM vehicles WHERE id = ?', [cleanVehicleId]);
      if (vehicleExists.length === 0) {
        return res.status(400).json({ message: 'Vehicle not found' });
      }
    }

    // Calculate total from quantity × rate
    const calculatedTotal = parseFloat(quantity) * parseFloat(rate);
    
    // Handle advance and balance
    const advancePaid = parseFloat(advance_paid) || 0;
    const finalBalance = parseFloat(balance) || (calculatedTotal - advancePaid);
    
    // Check if pdf and file_type columns exist
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'invoices'
      AND COLUMN_NAME IN ('pdf', 'file_type')
    `);
    
    const hasPdfColumn = columns.some(col => col.COLUMN_NAME === 'pdf');
    const hasFileTypeColumn = columns.some(col => col.COLUMN_NAME === 'file_type');
    
    let insertQuery, insertParams;
    
    if (hasPdfColumn && hasFileTypeColumn) {
      // Full query with pdf and file_type columns
      insertQuery = `
        INSERT INTO invoices (
          supplier_invoice_number, supplier_invoice_date,
          company_id, customer_id, from_location, destination, quantity, rate, amount, total,
          advance_paid, balance, driver_id, vehicle_id, driver_charges,
          image, pdf, file_type, invoice_date
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;
      insertParams = [
        supplier_invoice_number, supplier_invoice_date,
        company_id, customer_id, from_location, destination, quantity, rate, calculatedTotal, calculatedTotal,
        advancePaid, finalBalance, cleanDriverId, cleanVehicleId, JSON.stringify(parsedDriverCharges),
        imageBuffer, pdfBuffer, fileType, supplier_invoice_date
      ];
    } else if (hasFileTypeColumn) {
      // Query without pdf column
      insertQuery = `
        INSERT INTO invoices (
          supplier_invoice_number, supplier_invoice_date,
          company_id, customer_id, from_location, destination, quantity, rate, amount, total,
          advance_paid, balance, driver_id, vehicle_id, driver_charges,
          image, file_type, invoice_date
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;
      insertParams = [
        supplier_invoice_number, supplier_invoice_date,
        company_id, customer_id, from_location, destination, quantity, rate, calculatedTotal, calculatedTotal,
        advancePaid, finalBalance, cleanDriverId, cleanVehicleId, JSON.stringify(parsedDriverCharges),
        imageBuffer, fileType, supplier_invoice_date
      ];
    } else {
      // Query without pdf and file_type columns
      insertQuery = `
      INSERT INTO invoices (
        supplier_invoice_number, supplier_invoice_date,
        company_id, customer_id, from_location, destination, quantity, rate, amount, total,
        advance_paid, balance, driver_id, vehicle_id, driver_charges,
        image, invoice_date
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;
      insertParams = [
      supplier_invoice_number, supplier_invoice_date,
      company_id, customer_id, from_location, destination, quantity, rate, calculatedTotal, calculatedTotal,
      advancePaid, finalBalance, cleanDriverId, cleanVehicleId, JSON.stringify(parsedDriverCharges),
      imageBuffer, supplier_invoice_date
      ];
    }
    
    const [result] = await db.query(insertQuery, insertParams);
    
    res.status(201).json({ 
      message: 'Invoice created successfully',
      id: result.insertId
    });
  } catch (err) {
    console.error('Error creating invoice:', err);
    
    // Provide more specific error messages
    if (err.code === 'ER_NO_REFERENCED_ROW_2') {
      if (err.message.includes('vehicle_id')) {
        return res.status(400).json({ 
          message: 'Invalid vehicle selected. Please select a valid vehicle or leave it empty.',
          error: 'Vehicle not found'
        });
      } else if (err.message.includes('driver_id')) {
        return res.status(400).json({ 
          message: 'Invalid driver selected. Please select a valid driver or leave it empty.',
          error: 'Driver not found'
        });
      } else if (err.message.includes('company_id')) {
        return res.status(400).json({ 
          message: 'Invalid company selected. Please select a valid company.',
          error: 'Company not found'
        });
      } else if (err.message.includes('customer_id')) {
        return res.status(400).json({ 
          message: 'Invalid customer selected. Please select a valid customer.',
          error: 'Customer not found'
        });
      }
    }
    
    res.status(500).json({ 
      message: 'Server error. Please try again.', 
      error: err.message,
      code: err.code 
    });
  }
});

// Update invoice
router.put('/:id', upload.fields([
  { name: 'image', maxCount: 1 },
  { name: 'pdf', maxCount: 1 }
]), async (req, res) => {
  const { id } = req.params;
  const {
    company_id,
    customer_id,
    from_location,
    destination,
    supplier_invoice_date,
    quantity,
    rate,
    advance_paid,
    balance,
    driver_id,
    vehicle_id,
    supplier_invoice_number,
    driver_charges
  } = req.body;
  
  // Parse driver_charges if it's a string (from FormData)
  let parsedDriverCharges = [];
  if (driver_charges) {
    try {
      parsedDriverCharges = typeof driver_charges === 'string' ? JSON.parse(driver_charges) : driver_charges;
    } catch (error) {
      console.error('Error parsing driver_charges:', error);
      parsedDriverCharges = [];
    }
  }
  
  let imageBuffer = null;
  let pdfBuffer = null;
  let fileType = null;
  
  // Handle image file
  if (req.files && req.files.image && req.files.image[0]) {
    imageBuffer = req.files.image[0].buffer;
    fileType = 'image';
    console.log('Image file received for update:', req.files.image[0].originalname);
  }
  
  // Handle PDF file
  if (req.files && req.files.pdf && req.files.pdf[0]) {
    pdfBuffer = req.files.pdf[0].buffer;
    fileType = 'pdf';
    console.log('PDF file received for update:', req.files.pdf[0].originalname);
  }

  // Convert empty strings to null for optional fields
  const cleanDriverId = driver_id && driver_id.trim() !== '' ? driver_id : null;
  const cleanVehicleId = vehicle_id && vehicle_id.trim() !== '' ? vehicle_id : null;

  // Validation
  if (!company_id || !customer_id || !from_location || !destination || !supplier_invoice_date || 
      !quantity || !rate || !supplier_invoice_number) {
    return res.status(400).json({ 
      message: 'Company, customer, from location, destination, supplier invoice date, quantity, rate, and supplier invoice number are required' 
    });
  }

  try {
    // Check if invoice exists
    const [existing] = await db.query('SELECT id FROM invoices WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    // Check if supplier invoice number already exists (excluding current invoice)
    const [existingSupplierInvoice] = await db.query(
      'SELECT id FROM invoices WHERE supplier_invoice_number = ? AND id != ?', 
      [supplier_invoice_number, id]
    );
    if (existingSupplierInvoice.length > 0) {
      return res.status(400).json({ message: 'Supplier invoice number already exists' });
    }

    // Check if company exists
    console.log('🔍 Invoice Update - Validating company_id:', company_id, 'type:', typeof company_id);
    const [companyExists] = await db.query('SELECT id, company_name FROM company_details WHERE id = ?', [company_id]);
    console.log('📋 Invoice Update - Company check result:', companyExists);
    
    if (companyExists.length === 0) {
      // Get available companies for debugging
      const [availableCompanies] = await db.query('SELECT id, company_name FROM company_details ORDER BY id DESC LIMIT 10');
      console.log('📋 Available companies:', availableCompanies);
      
      return res.status(400).json({ 
        message: 'Company not found',
        company_id: company_id,
        company_id_type: typeof company_id,
        available_companies: availableCompanies
      });
    }
    console.log('✅ Invoice Update - Company validation passed:', companyExists[0]);

    // Check if customer exists
    const [customerExists] = await db.query('SELECT id FROM customers WHERE id = ?', [customer_id]);
    if (customerExists.length === 0) {
      return res.status(400).json({ message: 'Customer not found' });
    }

    // Check if driver exists (if provided)
    if (cleanDriverId) {
      const [driverExists] = await db.query('SELECT id FROM drivers WHERE id = ?', [cleanDriverId]);
      if (driverExists.length === 0) {
        return res.status(400).json({ message: 'Driver not found' });
      }
    }

    // Check if vehicle exists (if provided)
    if (cleanVehicleId) {
      const [vehicleExists] = await db.query('SELECT id FROM vehicles WHERE id = ?', [cleanVehicleId]);
      if (vehicleExists.length === 0) {
        return res.status(400).json({ message: 'Vehicle not found' });
      }
    }

    // Calculate total from quantity × rate
    const calculatedTotal = parseFloat(quantity) * parseFloat(rate);
    
    // Handle advance and balance
    const advancePaid = parseFloat(advance_paid) || 0;
    const finalBalance = parseFloat(balance) || (calculatedTotal - advancePaid);
    
    // Check if pdf and file_type columns exist
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'invoices'
      AND COLUMN_NAME IN ('pdf', 'file_type')
    `);
    
    const hasPdfColumn = columns.some(col => col.COLUMN_NAME === 'pdf');
    const hasFileTypeColumn = columns.some(col => col.COLUMN_NAME === 'file_type');
    
    let updateQuery = `
      UPDATE invoices SET 
        supplier_invoice_number = ?, supplier_invoice_date = ?,
        company_id = ?, customer_id = ?, from_location = ?, destination = ?, quantity = ?, rate = ?, amount = ?, total = ?,
        advance_paid = ?, balance = ?, driver_id = ?, vehicle_id = ?, driver_charges = ?`;
    let params = [
      supplier_invoice_number, supplier_invoice_date,
      company_id, customer_id, from_location, destination, quantity, rate, calculatedTotal, calculatedTotal,
      advancePaid, finalBalance, cleanDriverId, cleanVehicleId, JSON.stringify(parsedDriverCharges)
    ];
    
    if (hasPdfColumn && hasFileTypeColumn) {
      if (imageBuffer) {
        updateQuery += ', image = ?, pdf = NULL, file_type = ?';
        params.push(imageBuffer, 'image');
      } else if (pdfBuffer) {
        updateQuery += ', image = NULL, pdf = ?, file_type = ?';
        params.push(pdfBuffer, 'pdf');
      }
    } else if (hasFileTypeColumn) {
      if (imageBuffer) {
        updateQuery += ', image = ?, file_type = ?';
        params.push(imageBuffer, 'image');
      } else if (pdfBuffer) {
        updateQuery += ', file_type = ?';
        params.push('pdf');
      }
    } else {
    if (imageBuffer) {
      updateQuery += ', image = ?';
      params.push(imageBuffer);
    }
    }
    
    updateQuery += ' WHERE id = ?';
    params.push(id);
    await db.query(updateQuery, params);
    
    res.json({ message: 'Invoice updated successfully' });
  } catch (err) {
    console.error('Error updating invoice:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete invoice
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    // Check if invoice exists
    const [existing] = await db.query('SELECT id FROM invoices WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Invoice not found' });
    }

    await db.query('DELETE FROM invoices WHERE id = ?', [id]);
    res.json({ message: 'Invoice deleted successfully' });
  } catch (err) {
    console.error('Error deleting invoice:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update invoice status
router.patch('/:id/status', async (req, res) => {
  const { id } = req.params;
  const { status, paid_date } = req.body;
  
  if (!status || !['pending', 'sent', 'paid', 'cancelled'].includes(status)) {
    return res.status(400).json({ message: 'Valid status (pending, sent, paid, cancelled) is required' });
  }

  try {
    // If status is being set to 'paid', require paid_date
    if (status === 'paid' && !paid_date) {
      return res.status(400).json({ message: 'Paid date is required when setting status to paid' });
    }

    let query, params;
    
    if (status === 'paid') {
      query = 'UPDATE invoices SET status = ?, paid_date = ? WHERE id = ?';
      params = [status, paid_date, id];
    } else {
      query = 'UPDATE invoices SET status = ?, paid_date = NULL WHERE id = ?';
      params = [status, id];
    }

    const [result] = await db.query(query, params);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Invoice not found' });
    }
    
    res.json({ message: 'Invoice status updated successfully' });
  } catch (err) {
    console.error('Error updating invoice status:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add a charge (fuel, toll, driver) to an invoice
router.patch('/:id/add-charge', async (req, res) => {
  const { id } = req.params;
  const { type, value } = req.body;
  if (!['fuel', 'toll', 'driver'].includes(type)) {
    return res.status(400).json({ message: 'Invalid charge type' });
  }
  try {
    // Fetch current charges
    const [rows] = await db.query('SELECT driver_charges FROM invoices WHERE id = ?', [id]);
    if (rows.length === 0) return res.status(404).json({ message: 'Invoice not found' });
    
    let driverCharges = [];
    try {
      driverCharges = rows[0].driver_charges ? JSON.parse(rows[0].driver_charges) : [];
    } catch (error) {
      console.error('Error parsing driver_charges:', error);
      driverCharges = [];
    }
    
    // Add new charge with type information
    const newCharge = {
      type: type,
      amount: value,
      date: new Date().toISOString().slice(0, 10)
    };
    
    driverCharges.push(newCharge);
    
    await db.query('UPDATE invoices SET driver_charges = ? WHERE id = ?', [
      JSON.stringify(driverCharges),
      id
    ]);
    
    res.json({ message: 'Charge added successfully' });
  } catch (err) {
    console.error('Error adding charge:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Generate Bill for selected invoices
router.patch('/generate-bill', async (req, res) => {
  const { invoiceIds, bill_no, bill_created_at, settings_id, company_id } = req.body;
  
  if (!Array.isArray(invoiceIds) || invoiceIds.length === 0) {
    return res.status(400).json({ message: 'Invoice IDs array is required' });
  }
  
  if (!bill_no || bill_no.trim() === '') {
    return res.status(400).json({ message: 'Bill number is required' });
  }
  
  if (!settings_id) {
    return res.status(400).json({ message: 'Company details (settings_id) is required' });
  }
  
  if (!company_id) {
    return res.status(400).json({ message: 'Company ID is required' });
  }
  
  try {
    // Use the provided bill number
    const finalBillNo = bill_no.trim();
    
    // Use provided bill_created_at or current date
    const finalBillCreatedAt = bill_created_at || new Date().toISOString().slice(0, 10);
    
    // Update all selected invoices
    const [result] = await db.query(
      `UPDATE invoices SET bill_no = ?, bill_created_at = ?, status = 'sent', settings_id = ?, company_id = ? WHERE id IN (${invoiceIds.map(() => '?').join(',')})`,
      [finalBillNo, finalBillCreatedAt, settings_id, company_id, ...invoiceIds]
    );
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'No invoices found to update' });
    }
    
    res.json({ 
      success: true, 
      updated: result.affectedRows,
      bill_no: finalBillNo,
      bill_created_at: finalBillCreatedAt,
      settings_id: settings_id,
      company_id: company_id
    });
  } catch (err) {
    console.error('Error generating bill:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Serve invoice image as a file
router.get('/:id/image', async (req, res) => {
  const { id } = req.params;
  try {
    const [rows] = await db.query('SELECT image FROM invoices WHERE id = ?', [id]);
    if (rows.length === 0 || !rows[0].image) {
      return res.status(404).send('No image found');
    }
    const imageBuffer = rows[0].image;
    // Try to detect type (default to jpeg)
    let type = 'image/jpeg';
    // Simple check for PNG signature
    if (imageBuffer && imageBuffer.length > 4 && imageBuffer[0] === 0x89 && imageBuffer[1] === 0x50) {
      type = 'image/png';
    }
    res.set('Content-Type', type);
    res.send(imageBuffer);
  } catch (err) {
    res.status(500).send('Server error');
  }
});

// Serve invoice PDF as a file
router.get('/:id/pdf', async (req, res) => {
  const { id } = req.params;
  try {
    // First check if pdf column exists
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'invoices'
      AND COLUMN_NAME = 'pdf'
    `);
    
    if (columns.length === 0) {
      return res.status(404).send('PDF column not found in database');
    }
    
    const [rows] = await db.query('SELECT pdf FROM invoices WHERE id = ?', [id]);
    if (rows.length === 0 || !rows[0].pdf) {
      return res.status(404).send('No PDF found');
    }
    const pdfBuffer = rows[0].pdf;
    res.set('Content-Type', 'application/pdf');
    res.set('Content-Disposition', `inline; filename="invoice-${id}.pdf"`);
    res.send(pdfBuffer);
  } catch (err) {
    console.error('Error serving PDF:', err);
    res.status(500).send('Server error');
  }
});

// Get all invoices for a given vehicle_id
router.get('/by-vehicle/:vehicleId', async (req, res) => {
  const { vehicleId } = req.params;
  try {
    console.log('Fetching invoices for vehicle ID:', vehicleId);
    const [rows] = await db.query(`
      SELECT 
        i.id,
        i.supplier_invoice_number,
        i.supplier_invoice_date,
        i.invoice_date,
        i.company_id,
        i.customer_id,
        i.from_location,
        i.destination,
        i.quantity,
        i.rate,
        i.amount,
        i.total,
        i.advance_paid,
        i.balance,
        i.driver_id,
        i.vehicle_id,
        i.driver_charges,
        i.status,
        i.created_at,
        i.bill_no,
        i.bill_created_at,
        i.settings_id,
        i.paid_date,
        c.name as customer_name,
        c.contact as customer_contact,
        c.location as customer_location,
        d.name as driver_name,
        d.phone as driver_phone,
        v.number as vehicle_number,
        v.name as vehicle_name,
        co.name as company_name
      FROM invoices i
      LEFT JOIN customers c ON i.customer_id = c.id
      LEFT JOIN drivers d ON i.driver_id = d.id
      LEFT JOIN vehicles v ON i.vehicle_id = v.id
      LEFT JOIN companies co ON i.company_id = co.id
      WHERE i.vehicle_id = ?
      ORDER BY i.created_at DESC
    `, [vehicleId]);
    
    console.log('Found invoices:', rows.length);
    // Parse JSON fields and handle dates properly - same as main route
    const invoices = rows.map(invoice => {
      // Convert created_at to Indian time
      const createdAt = new Date(invoice.created_at);
      const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
      
      // Handle supplier_invoice_date - return exactly as stored in database (no timezone conversion)
      let supplierInvoiceDate = invoice.supplier_invoice_date;
      if (supplierInvoiceDate) {
        // If it's a Date object, format it as YYYY-MM-DD without timezone conversion
        if (supplierInvoiceDate instanceof Date) {
          const year = supplierInvoiceDate.getFullYear();
          const month = String(supplierInvoiceDate.getMonth() + 1).padStart(2, '0');
          const day = String(supplierInvoiceDate.getDate()).padStart(2, '0');
          supplierInvoiceDate = `${year}-${month}-${day}`;
        } else if (typeof supplierInvoiceDate === 'string') {
          // If it's already a string, ensure it's in YYYY-MM-DD format
          supplierInvoiceDate = supplierInvoiceDate.slice(0, 10);
        }
      }
      
      // Safely parse driver_charges
      let driverCharges = [];
      try {
        driverCharges = invoice.driver_charges ? JSON.parse(invoice.driver_charges) : [];
      } catch (error) {
        console.error('Error parsing driver_charges for invoice', invoice.id, ':', error);
        driverCharges = [];
      }
      
      return {
        ...invoice,
        supplier_invoice_date: supplierInvoiceDate,
        created_at: indianCreatedAt.toISOString(),
        driver_charges: driverCharges
      };
    });
    
    res.json(invoices);
  } catch (err) {
    console.error('Error fetching invoices by vehicle:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get all invoices for a given driver_id
router.get('/by-driver/:driverId', async (req, res) => {
  const { driverId } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        i.id,
        i.supplier_invoice_number,
        DATE_FORMAT(i.supplier_invoice_date, '%d/%m/%Y') as supplier_invoice_date_formatted,
        i.company_id,
        i.customer_id,
        i.from_location,
        i.destination,
        i.quantity,
        i.rate,
        i.amount,
        i.total,
        i.advance_paid,
        i.balance,
        i.driver_id,
        i.vehicle_id,
        i.driver_charges,
        i.status,
        i.created_at,
        i.bill_no,
        i.bill_created_at,
        i.settings_id,
        i.paid_date,
        c.name as customer_name,
        c.contact as customer_contact,
        c.location as customer_location,
        d.name as driver_name,
        d.phone as driver_phone,
        v.number as vehicle_number,
        v.name as vehicle_name,
        co.name as company_name
      FROM invoices i
      LEFT JOIN customers c ON i.customer_id = c.id
      LEFT JOIN drivers d ON i.driver_id = d.id
      LEFT JOIN vehicles v ON i.vehicle_id = v.id
      LEFT JOIN companies co ON i.company_id = co.id
      WHERE i.driver_id = ?
      ORDER BY i.created_at DESC
    `, [driverId]);
    
    // Parse JSON fields and handle dates properly - same as main route
    const invoices = rows.map(invoice => {
      // Convert created_at to Indian time
      const createdAt = new Date(invoice.created_at);
      const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
      
      // Use the formatted date from database query
      const supplierInvoiceDate = invoice.supplier_invoice_date_formatted || invoice.supplier_invoice_date;
      
      // Safely parse driver_charges
      let driverCharges = [];
      try {
        driverCharges = invoice.driver_charges ? JSON.parse(invoice.driver_charges) : [];
      } catch (error) {
        console.error('Error parsing driver_charges for invoice', invoice.id, ':', error);
        driverCharges = [];
      }
      
      return {
        ...invoice,
        supplier_invoice_date: supplierInvoiceDate,
        supplier_invoice_date_formatted: invoice.supplier_invoice_date_formatted,
        created_at: indianCreatedAt.toISOString(),
        driver_charges: driverCharges
      };
    });
    
    res.json(invoices);
  } catch (err) {
    console.error('Error fetching invoices by driver:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get all invoices for a given company_id
router.get('/by-company/:companyId', async (req, res) => {
  const { companyId } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        i.id,
        i.supplier_invoice_number,
        i.supplier_invoice_date,
        i.invoice_date,
        i.company_id,
        i.customer_id,
        i.from_location,
        i.destination,
        i.quantity,
        i.rate,
        i.amount,
        i.total,
        i.advance_paid,
        i.balance,
        i.driver_id,
        i.vehicle_id,
        i.driver_charges,
        i.status,
        i.created_at,
        i.bill_no,
        i.bill_created_at,
        i.settings_id,
        i.paid_date,
        c.name as customer_name,
        c.contact as customer_contact,
        c.location as customer_location,
        d.name as driver_name,
        d.phone as driver_phone,
        v.number as vehicle_number,
        v.name as vehicle_name,
        co.name as company_name
      FROM invoices i
      LEFT JOIN customers c ON i.customer_id = c.id
      LEFT JOIN drivers d ON i.driver_id = d.id
      LEFT JOIN vehicles v ON i.vehicle_id = v.id
      LEFT JOIN companies co ON i.company_id = co.id
      WHERE i.company_id = ?
      ORDER BY i.created_at DESC
    `, [companyId]);
    
    // Parse JSON fields and handle dates properly - same as main route
    const invoices = rows.map(invoice => {
      // Convert created_at to Indian time
      const createdAt = new Date(invoice.created_at);
      const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
      
      // Handle supplier_invoice_date - return exactly as stored in database (no timezone conversion)
      let supplierInvoiceDate = invoice.supplier_invoice_date;
      if (supplierInvoiceDate) {
        // If it's a Date object, format it as YYYY-MM-DD without timezone conversion
        if (supplierInvoiceDate instanceof Date) {
          const year = supplierInvoiceDate.getFullYear();
          const month = String(supplierInvoiceDate.getMonth() + 1).padStart(2, '0');
          const day = String(supplierInvoiceDate.getDate()).padStart(2, '0');
          supplierInvoiceDate = `${year}-${month}-${day}`;
        } else if (typeof supplierInvoiceDate === 'string') {
          // If it's already a string, ensure it's in YYYY-MM-DD format
          supplierInvoiceDate = supplierInvoiceDate.slice(0, 10);
        }
      }
      
      // Safely parse driver_charges
      let driverCharges = [];
      try {
        driverCharges = invoice.driver_charges ? JSON.parse(invoice.driver_charges) : [];
      } catch (error) {
        console.error('Error parsing driver_charges for invoice', invoice.id, ':', error);
        driverCharges = [];
      }
      
      return {
        ...invoice,
        supplier_invoice_date: supplierInvoiceDate,
        created_at: indianCreatedAt.toISOString(),
        driver_charges: driverCharges
      };
    });
    
    res.json(invoices);
  } catch (err) {
    console.error('Error fetching invoices by company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 