const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// Get all receipts
router.get('/', async (req, res) => {
  try {
    // Check what columns actually exist in the receipts table
    const [tableInfo] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'receipts'
    `);
    
    const existingColumns = tableInfo.map(row => row.COLUMN_NAME);
    console.log('Existing columns in receipts table:', existingColumns);
    
    // Build dynamic SELECT query based on existing columns
    let selectFields = ['r.id', 'r.created_at'];
    let joins = [];
    
    if (existingColumns.includes('company_id')) {
      selectFields.push('r.company_id');
      joins.push('LEFT JOIN companies c ON r.company_id = c.id');
      selectFields.push('c.name as company_name');
    }
    
    if (existingColumns.includes('bill_no')) {
      selectFields.push('r.bill_no');
    }
    
    if (existingColumns.includes('payment_date')) {
      selectFields.push('r.payment_date');
    }
    
    if (existingColumns.includes('payment_amount')) {
      selectFields.push('r.payment_amount');
    }
    
    if (existingColumns.includes('payment_mode')) {
      selectFields.push('r.payment_mode');
    }
    
    if (existingColumns.includes('reference_number')) {
      selectFields.push('r.reference_number');
    }
    
    if (existingColumns.includes('notes')) {
      selectFields.push('r.notes');
    }
    
    if (existingColumns.includes('bank_id')) {
      selectFields.push('r.bank_id');
      joins.push('LEFT JOIN banks b ON r.bank_id = b.id');
      selectFields.push('b.bank_name', 'b.account_number', 'b.ifsc_code', 'b.branch', 'b.account_name as bank_account_name');
    }
    
    if (existingColumns.includes('bunk_id')) {
      selectFields.push('r.bunk_id');
      joins.push('LEFT JOIN bunks bk ON r.bunk_id = bk.id');
      selectFields.push('bk.bunk_name');
    }
    
    if (existingColumns.includes('customer_id')) {
      selectFields.push('r.customer_id');
      joins.push('LEFT JOIN customers cust ON r.customer_id = cust.id');
      selectFields.push('cust.name as customer_name');
    }
    
    if (existingColumns.includes('driver_id')) {
      selectFields.push('r.driver_id');
      joins.push('LEFT JOIN drivers d ON r.driver_id = d.id');
      selectFields.push('d.name as driver_name');
    }
    
    if (existingColumns.includes('vehicle_no')) {
      selectFields.push('r.vehicle_no');
    }
    
    const selectQuery = `
      SELECT ${selectFields.join(', ')}
      FROM receipts r
      ${joins.join(' ')}
      ORDER BY r.created_at DESC, r.id DESC
    `;
    
    console.log('Select query:', selectQuery);
    
    const [rows] = await db.query(selectQuery);
    
    console.log('Debug - Fetched receipts:', rows.length);
    if (rows.length > 0) {
      console.log('Debug - Sample receipt:', {
        id: rows[0].id,
        customer_id: rows[0].customer_id,
        customer_name: rows[0].customer_name,
        bunk_id: rows[0].bunk_id,
        bunk_name: rows[0].bunk_name,
        bank_id: rows[0].bank_id,
        bank_name: rows[0].bank_name
      });
    }
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching receipts:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get receipts by company
router.get('/company/:companyId', async (req, res) => {
  const { companyId } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        r.id,
        r.company_id,
        r.bill_no,
        r.payment_date,
        r.payment_amount,
        r.payment_mode,
        r.reference_number,
        r.notes,
        r.bank_id,
        r.created_at,
        c.name as company_name,
        b.bank_name,
        b.account_number,
        b.ifsc_code,
        b.branch,
        b.account_name as bank_account_name,
        DATE_FORMAT(r.payment_date, '%Y-%m-%d') as formatted_payment_date
      FROM receipts r
      LEFT JOIN companies c ON r.company_id = c.id
      LEFT JOIN banks b ON r.bank_id = b.id
      WHERE r.company_id = ?
      ORDER BY r.payment_date DESC, r.created_at DESC
    `, [companyId]);
    
    // Format dates consistently
    const formattedRows = rows.map(row => ({
      ...row,
      payment_date: row.formatted_payment_date || row.payment_date,
      created_at: row.created_at ? new Date(row.created_at).toISOString().split('T')[0] : null
    }));
    
    res.json(formattedRows);
  } catch (err) {
    console.error('Error fetching receipts by company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get receipts by driver ID (direct relationship and through company relationships)
router.get('/by-driver/:driverId', async (req, res) => {
  const { driverId } = req.params;
  try {
    console.log('Fetching receipts for driver ID:', driverId);
    
    // Get receipts that are directly related to this driver (driver_id field only)
    const [rows] = await db.query(`
      SELECT DISTINCT
        r.id,
        r.company_id,
        r.driver_id,
        r.bill_no,
        DATE_FORMAT(r.payment_date, '%Y-%m-%d') as payment_date_formatted,
        r.payment_amount,
        r.payment_mode,
        r.reference_number,
        r.notes,
        r.bank_id,
        r.created_at,
        c.name as company_name,
        b.bank_name,
        b.account_number,
        b.ifsc_code,
        b.branch,
        b.account_name as bank_account_name,
        'receipt' as transaction_type
      FROM receipts r
      LEFT JOIN companies c ON r.company_id = c.id
      LEFT JOIN banks b ON r.bank_id = b.id
      WHERE r.driver_id = ?
      ORDER BY r.payment_date DESC, r.created_at DESC
    `, [driverId]);
    
    // Ensure consistent date format
    const formattedRows = rows.map(row => ({
      ...row,
      payment_date: row.payment_date_formatted || row.payment_date
    }));
    
    console.log(`Fetched ${formattedRows.length} receipts for driver ${driverId}:`, formattedRows.map(r => ({
      id: r.id,
      bill_no: r.bill_no,
      company_id: r.company_id,
      driver_id: r.driver_id,
      company_name: r.company_name,
      payment_amount: r.payment_amount,
      payment_date: r.payment_date
    })));
    
    // Also check what receipts exist in the database
    const [allReceipts] = await db.query(`
      SELECT id, bill_no, driver_id, company_id, payment_amount
      FROM receipts 
      ORDER BY created_at DESC 
      LIMIT 10
    `);
    console.log('Recent receipts in database:', allReceipts);
    
    res.json(formattedRows);
  } catch (err) {
    console.error('Error fetching receipts by driver:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get company balance summary
router.get('/balance/:companyId', async (req, res) => {
  const { companyId } = req.params;
  try {
    // Get all sent invoices for the company
    const [invoices] = await db.query(`
      SELECT 
        bill_no,
        bill_created_at,
        SUM(total) as bill_amount
      FROM invoices 
      WHERE company_id = ? AND status = 'sent' AND bill_no IS NOT NULL
      GROUP BY bill_no, bill_created_at
      ORDER BY bill_created_at ASC
    `, [companyId]);

    // Get all receipts for the company
    const [receipts] = await db.query(`
      SELECT 
        payment_date,
        payment_amount,
        bill_no
      FROM receipts 
      WHERE company_id = ?
      ORDER BY payment_date ASC
    `, [companyId]);

    // Calculate running balance
    let totalBilled = 0;
    let totalPaid = 0;
    let balance = 0;
    
    const balanceHistory = [];

    // Add all bills first
    invoices.forEach(bill => {
      totalBilled += parseFloat(bill.bill_amount);
      balance += parseFloat(bill.bill_amount);
      balanceHistory.push({
        date: bill.bill_created_at,
        type: 'bill',
        bill_no: bill.bill_no,
        amount: parseFloat(bill.bill_amount),
        balance: balance,
        description: `Bill ${bill.bill_no}`
      });
    });

    // Add all payments
    receipts.forEach(payment => {
      totalPaid += parseFloat(payment.payment_amount);
      balance -= parseFloat(payment.payment_amount);
      balanceHistory.push({
        date: payment.payment_date,
        type: 'payment',
        bill_no: payment.bill_no,
        amount: -parseFloat(payment.payment_amount),
        balance: balance,
        description: `Payment ${payment.bill_no ? `for ${payment.bill_no}` : ''}`
      });
    });

    // Sort by date
    balanceHistory.sort((a, b) => new Date(a.date) - new Date(b.date));

    res.json({
      company_id: companyId,
      total_billed: totalBilled,
      total_paid: totalPaid,
      current_balance: balance,
      balance_history: balanceHistory
    });
  } catch (err) {
    console.error('Error calculating company balance:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get sent bills for a company
router.get('/sent-bills/:companyId', async (req, res) => {
  const { companyId } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        bill_no,
        bill_created_at,
        COUNT(*) as invoice_count,
        SUM(total) as total_amount,
        SUM(quantity) as total_quantity
      FROM invoices 
      WHERE company_id = ? AND status = 'sent' AND bill_no IS NOT NULL
      GROUP BY bill_no, bill_created_at
      ORDER BY bill_created_at DESC
    `, [companyId]);
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching sent bills:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Create new receipt
router.post('/', async (req, res) => {
  const {
    company_id,
    bill_no,
    payment_date,
    payment_amount,
    payment_mode,
    reference_number,
    notes,
    bank_id,
    bunk_id,
    customer_id,
    driver_id,
    vehicle_no
  } = req.body;
  
  console.log('🔍 Receipt creation - Raw request body:', req.body);
  console.log('🔍 Receipt creation - Extracted values:', {
    company_id,
    customer_id,
    bank_id,
    bunk_id,
    driver_id,
    vehicle_no,
    bill_no,
    payment_amount
  });
  
  console.log('Debug - Receipt creation request body:', req.body);
  console.log('Debug - Extracted bank_id:', bank_id);
  console.log('Debug - Payment mode:', payment_mode);
  
  // Validation
  if (!payment_date || !payment_amount) {
    return res.status(400).json({ 
      message: 'Payment date and payment amount are required' 
    });
  }

  try {
    // Ensure required columns exist in receipts table
    try {
      await db.query('ALTER TABLE receipts ADD COLUMN customer_id INT NULL');
      console.log('✅ Added customer_id column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await db.query('ALTER TABLE receipts ADD COLUMN bunk_id INT NULL');
      console.log('✅ Added bunk_id column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await db.query('ALTER TABLE receipts ADD COLUMN bank_id INT NULL');
      console.log('✅ Added bank_id column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await db.query('ALTER TABLE receipts ADD COLUMN driver_id INT NULL');
      console.log('✅ Added driver_id column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await db.query('ALTER TABLE receipts ADD COLUMN vehicle_no VARCHAR(255) NULL');
      console.log('✅ Added vehicle_no column');
    } catch (error) {
      // Column already exists
    }

    // Check if company exists only if company_id is provided
    if (company_id) {
      const [companyExists] = await db.query('SELECT id FROM company_details WHERE id = ?', [company_id]);
      if (companyExists.length === 0) {
        return res.status(400).json({ message: 'Company not found' });
      }
    }

    // Check if company_id column allows NULL
    const [columnInfo] = await db.query(`
      SELECT IS_NULLABLE 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'receipts'
      AND COLUMN_NAME = 'company_id'
    `);

    const allowsNull = columnInfo.length > 0 && columnInfo[0].IS_NULLABLE === 'YES';
    console.log('🔍 company_id column allows NULL:', allowsNull);

    // Check if bank exists if bank_id is provided
    if (bank_id) {
      const [bankExists] = await db.query('SELECT id FROM banks WHERE id = ?', [bank_id]);
      if (bankExists.length === 0) {
        return res.status(400).json({ message: 'Bank not found' });
      }
      console.log('Debug - Bank found:', bankExists[0]);
    } else {
      console.log('Debug - No bank_id provided');
    }

    // Note: vehicle_id is not used in receipts table

    // Check if bunk exists if bunk_id is provided
    if (bunk_id) {
      const [bunkExists] = await db.query('SELECT id FROM bunks WHERE id = ?', [bunk_id]);
      if (bunkExists.length === 0) {
        return res.status(400).json({ message: 'Bunk not found' });
      }
      console.log('Debug - Bunk found:', bunkExists[0]);
    } else {
      console.log('Debug - No bunk_id provided');
    }

    // Check if customer exists if customer_id is provided
    if (customer_id) {
      const [customerExists] = await db.query('SELECT id FROM customers WHERE id = ?', [customer_id]);
      if (customerExists.length === 0) {
        return res.status(400).json({ message: 'Customer not found' });
      }
      console.log('Debug - Customer found:', customerExists[0]);
    } else {
      console.log('Debug - No customer_id provided');
    }

    // Check what columns actually exist in the receipts table
    const [tableInfo] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'receipts'
    `);
    
    const existingColumns = tableInfo.map(row => row.COLUMN_NAME);
    console.log('Existing columns in receipts table:', existingColumns);
    
    // Build dynamic INSERT query based on existing columns
    const insertFields = [];
    const insertValues = [];
    const placeholders = [];
    
    // Always include these basic fields
    if (existingColumns.includes('company_id')) {
      insertFields.push('company_id');
      // If column doesn't allow NULL and no company_id provided, use default company (ID 1)
      const finalCompanyId = (company_id || null) || (!allowsNull ? 1 : null);
      insertValues.push(finalCompanyId);
      placeholders.push('?');
      console.log('🔍 Final company_id for insert:', finalCompanyId, '(allowsNull:', allowsNull, ')');
    }
    
    if (existingColumns.includes('bill_no')) {
      insertFields.push('bill_no');
      insertValues.push(bill_no);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('payment_date')) {
      insertFields.push('payment_date');
      insertValues.push(payment_date);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('payment_amount')) {
      insertFields.push('payment_amount');
      insertValues.push(payment_amount);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('payment_mode')) {
      insertFields.push('payment_mode');
      insertValues.push(payment_mode);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('reference_number')) {
      insertFields.push('reference_number');
      insertValues.push(reference_number);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('notes')) {
      insertFields.push('notes');
      insertValues.push(notes);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('bank_id')) {
      insertFields.push('bank_id');
      insertValues.push(bank_id);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('bunk_id')) {
      insertFields.push('bunk_id');
      insertValues.push(bunk_id);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('customer_id')) {
      insertFields.push('customer_id');
      insertValues.push(customer_id);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('driver_id')) {
      insertFields.push('driver_id');
      insertValues.push(driver_id);
      placeholders.push('?');
    }
    
    if (existingColumns.includes('vehicle_no')) {
      insertFields.push('vehicle_no');
      insertValues.push(vehicle_no);
      placeholders.push('?');
    }
    
    const insertQuery = `
      INSERT INTO receipts (${insertFields.join(', ')})
      VALUES (${placeholders.join(', ')})
    `;
    
    console.log('🔍 Insert query:', insertQuery);
    console.log('🔍 Insert values:', insertValues);
    console.log('🔍 Field mapping:', insertFields.map((field, index) => `${field} = ${insertValues[index]}`));
    console.log('🔍 Customer ID being saved:', customer_id);
    console.log('🔍 Bunk ID being saved:', bunk_id);
    console.log('🔍 Bank ID being saved:', bank_id);
    console.log('🔍 Company ID being saved:', company_id);
    
    const [result] = await db.query(insertQuery, insertValues);
    
    console.log('Debug - Receipt created with ID:', result.insertId);
    console.log('Debug - Final saved values:', {
      customer_id,
      bunk_id,
      bank_id,
      company_id
    });
    
    res.status(201).json({ 
      message: 'Receipt created successfully',
      id: result.insertId
    });
  } catch (err) {
    console.error('Error creating receipt:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update receipt
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const {
    company_id,
    bill_no,
    payment_date,
    payment_amount,
    payment_mode,
    reference_number,
    notes,
    bank_id,
    bunk_id,
    customer_id
  } = req.body;
  
  // Validation
  if (!payment_date || !payment_amount) {
    return res.status(400).json({ 
      message: 'Payment date and payment amount are required' 
    });
  }

  try {
    // Check if receipt exists
    const [existing] = await db.query('SELECT id FROM receipts WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Receipt not found' });
    }

    // Check if company exists only if company_id is provided
    if (company_id) {
      const [companyExists] = await db.query('SELECT id FROM company_details WHERE id = ?', [company_id]);
      if (companyExists.length === 0) {
        return res.status(400).json({ message: 'Company not found' });
      }
    }

    // Check if bank exists if bank_id is provided
    if (bank_id) {
      const [bankExists] = await db.query('SELECT id FROM banks WHERE id = ?', [bank_id]);
      if (bankExists.length === 0) {
        return res.status(400).json({ message: 'Bank not found' });
      }
    }

    // Note: vehicle_id is not used in receipts table

    // Check if bunk exists if bunk_id is provided
    if (bunk_id) {
      const [bunkExists] = await db.query('SELECT id FROM bunks WHERE id = ?', [bunk_id]);
      if (bunkExists.length === 0) {
        return res.status(400).json({ message: 'Bunk not found' });
      }
    }

    // Check if customer exists if customer_id is provided
    if (customer_id) {
      const [customerExists] = await db.query('SELECT id FROM customers WHERE id = ?', [customer_id]);
      if (customerExists.length === 0) {
        return res.status(400).json({ message: 'Customer not found' });
      }
    }

    await db.query(`
      UPDATE receipts SET 
        company_id = ?, bill_no = ?, payment_date = ?, payment_amount = ?,
        payment_mode = ?, reference_number = ?, notes = ?, bank_id = ?, bunk_id = ?, customer_id = ?
      WHERE id = ?
    `, [company_id || null, bill_no, payment_date, payment_amount, payment_mode, reference_number, notes, bank_id, bunk_id || null, customer_id || null, id]);
    
    res.json({ message: 'Receipt updated successfully' });
  } catch (err) {
    console.error('Error updating receipt:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete receipt
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    // Check if receipt exists
    const [existing] = await db.query('SELECT id FROM receipts WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Receipt not found' });
    }

    await db.query('DELETE FROM receipts WHERE id = ?', [id]);
    res.json({ message: 'Receipt deleted successfully' });
  } catch (err) {
    console.error('Error deleting receipt:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get receipts by customer ID
router.get('/by-customer/:customerId', async (req, res) => {
  const { customerId } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        r.id,
        r.company_id,
        r.bill_no,
        DATE_FORMAT(r.payment_date, '%Y-%m-%d') as payment_date_formatted,
        r.payment_date,
        r.payment_amount,
        r.payment_mode,
        r.reference_number,
        r.notes,
        r.bank_id,
        r.bunk_id,
        r.customer_id,
        r.created_at,
        c.name as company_name,
        b.bank_name,
        b.account_number,
        b.ifsc_code,
        b.branch,
        b.account_name as bank_account_name,
        bk.bunk_name,
        cust.name as customer_name
      FROM receipts r
      LEFT JOIN companies c ON r.company_id = c.id
      LEFT JOIN banks b ON r.bank_id = b.id
      LEFT JOIN bunks bk ON r.bunk_id = bk.id
      LEFT JOIN customers cust ON r.customer_id = cust.id
      WHERE r.customer_id = ?
      ORDER BY r.payment_date DESC, r.created_at DESC
    `, [customerId]);
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching receipts by customer:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get receipts by vehicle number - Note: receipts table doesn't have vehicle_id
// This endpoint is disabled since receipts are not vehicle-specific
router.get('/by-vehicle/:vehicleNumber', async (req, res) => {
  res.json([]); // Return empty array since receipts don't have vehicle associations
});

// Get receipts by bunk ID
router.get('/by-bunk/:bunkId', async (req, res) => {
  const { bunkId } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        r.id,
        r.company_id,
        r.bill_no,
        DATE_FORMAT(r.payment_date, '%Y-%m-%d') as payment_date_formatted,
        r.payment_date,
        r.payment_amount,
        r.payment_mode,
        r.reference_number,
        r.notes,
        r.bank_id,
        r.bunk_id,
        r.created_at,
        c.name as company_name,
        b.bank_name,
        b.account_number,
        b.ifsc_code,
        b.branch,
        b.account_name as bank_account_name,
        bk.bunk_name
      FROM receipts r
      LEFT JOIN companies c ON r.company_id = c.id
      LEFT JOIN banks b ON r.bank_id = b.id
      LEFT JOIN bunks bk ON r.bunk_id = bk.id
      WHERE r.bunk_id = ?
      ORDER BY r.payment_date DESC, r.created_at DESC
    `, [bunkId]);
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching receipts by bunk:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 