const express = require('express');
const router = express.Router();

// Get transactions by bunk ID
router.get('/', async (req, res) => {
    try {
        const { bunkId } = req.query;
        
        if (!bunkId) {
            return res.status(400).json({ error: 'Bunk ID is required' });
        }

        // For now, return sample data structure
        // In a real implementation, you would query your database
        const sampleTransactions = [
            {
                id: 1,
                type: 'credit',
                amount: 5000,
                bill_no: 'R001',
                date: new Date().toISOString(),
                reference: 'Receipt Payment',
                bunk_id: parseInt(bunkId)
            },
            {
                id: 2,
                type: 'debit',
                amount: 2000,
                bill_no: 'P001',
                date: new Date(Date.now() - 86400000).toISOString(), // Yesterday
                reference: 'Payment Entry',
                bunk_id: parseInt(bunkId)
            },
            {
                id: 3,
                type: 'credit',
                amount: 3000,
                bill_no: 'R002',
                date: new Date(Date.now() - 172800000).toISOString(), // 2 days ago
                reference: 'Receipt Payment',
                bunk_id: parseInt(bunkId)
            }
        ];

        // Filter by bunk ID
        const bunkTransactions = sampleTransactions.filter(t => t.bunk_id === parseInt(bunkId));
        
        res.json(bunkTransactions);
    } catch (error) {
        console.error('Error fetching transactions:', error);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Get all transactions
router.get('/all', async (req, res) => {
    try {
        // Return all transactions for all bunks
        const allTransactions = [
            {
                id: 1,
                type: 'credit',
                amount: 5000,
                bill_no: 'R001',
                date: new Date().toISOString(),
                reference: 'Receipt Payment',
                bunk_id: 1
            },
            {
                id: 2,
                type: 'debit',
                amount: 2000,
                bill_no: 'P001',
                date: new Date(Date.now() - 86400000).toISOString(),
                reference: 'Payment Entry',
                bunk_id: 1
            },
            {
                id: 3,
                type: 'credit',
                amount: 3000,
                bill_no: 'R002',
                date: new Date(Date.now() - 172800000).toISOString(),
                reference: 'Receipt Payment',
                bunk_id: 2
            }
        ];
        
        res.json(allTransactions);
    } catch (error) {
        console.error('Error fetching all transactions:', error);
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Get transactions by vehicle ID
router.get('/vehicle/:vehicleId', async (req, res) => {
    try {
        const { vehicleId } = req.params;
        
        if (!vehicleId) {
            return res.status(400).json({ error: 'Vehicle ID is required' });
        }

        // For now, return sample data structure for vehicle transactions
        // In a real implementation, you would query your database for vehicle-specific transactions
        const sampleVehicleTransactions = [
            {
                id: 1,
                voucher_type: 'PAYMENT',
                bill_no: '1',
                date: new Date().toISOString().split('T')[0],
                payment_mode: 'CASH',
                expense_type: 'SALARY',
                name: 'ANIL',
                note: 'N/A',
                debit: 8000,
                credit: 0,
                balance: 0,
                vehicle_id: parseInt(vehicleId)
            },
            {
                id: 2,
                voucher_type: 'PAYMENT',
                bill_no: '5',
                date: new Date(Date.now() - 86400000).toISOString().split('T')[0], // Yesterday
                payment_mode: 'HDFC BANK',
                expense_type: 'DIESEL',
                name: 'AP39UB9380',
                note: 'N/A',
                debit: 1000,
                credit: 0,
                balance: 0,
                vehicle_id: parseInt(vehicleId)
            }
        ];

        // Filter by vehicle ID (in real implementation, this would be done in the database query)
        const vehicleTransactions = sampleVehicleTransactions.filter(t => t.vehicle_id === parseInt(vehicleId));
        
        res.json(vehicleTransactions);
    } catch (error) {
        console.error('Error fetching transactions by vehicle:', error);
        res.status(500).json({ error: 'Internal server error' });
    }
});

module.exports = router;
