const express = require('express');
  const cors = require('cors');
  const mysql = require('mysql2/promise');
  require('dotenv').config();
  const { router: authRouter, setDb: setAuthDb } = require('./routes/auth');
  const { router: vehiclesRouter, setDb: setVehiclesDb } = require('./routes/vehicles');
  const { router: customersRouter, setDb: setCustomersDb } = require('./routes/customers');
  const { router: driversRouter, setDb: setDriversDb } = require('./routes/drivers');
  const { router: invoicesRouter, setDb: setInvoicesDb } = require('./routes/invoices');
  const { router: settingsRouter, setDb: setSettingsDb } = require('./routes/settings');
  const { router: expensesRouter, setDb: setExpensesDb } = require('./routes/expenses');
  const { router: companyDetailsRouter, setDb: setCompanyDetailsDb } = require('./routes/company-details');
  const { router: dieselPurchaseRouter, setDb: setDieselPurchaseDb } = require('./routes/diesel-purchase');
  const { router: paymentEntryRouter, setDb: setPaymentEntryDb } = require('./routes/payment-entry');
  const { router: receiptsRouter, setDb: setReceiptsDb } = require('./routes/receipts');
  const { router: bunksRouter, setDb: setBunksDb } = require('./routes/bunks');
  const { router: banksRouter, setDb: setBanksDb } = require('./routes/banks');
  const { router: destinationsRouter, setDb: setDestinationsDb } = require('./routes/destinations');
  const transactionsRouter = require('./routes/transactions');
  const { router: expenseTypesRouter, setDb: setExpenseTypesDb } = require('./routes/expense-types');
  const companyRoutes = require('./routes/company');
  const billNumbersRouter = require('./routes/bill-numbers');
  const path = require('path');

  // Set timezone to Indian time
  process.env.TZ = 'Asia/Kolkata';

  const app = express();
  app.use(cors());
  app.use(express.json());
  app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

  // Health check endpoint
  app.get('/api/health', (req, res) => {
    res.json({ status: 'OK', message: 'Server is running' });
  });

  // Create MySQL connection pool
  const pool = mysql.createPool({
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'cement',
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0,
    timezone: '+05:30', // Indian timezone
  });

  // Test database connection
  pool.getConnection()
    .then(connection => {
      console.log('Database connected successfully');
      connection.release();
    })
    .catch(err => {
      console.error('Database connection error:', err);
    });

  // Set timezone for all connections
  pool.on('connection', function (connection) {
    connection.query('SET time_zone = "+05:30"');
  });

  // Set database connection for all routers
  setAuthDb(pool);
  setVehiclesDb(pool);
  setCustomersDb(pool);
  setDriversDb(pool);
  setInvoicesDb(pool);
  setSettingsDb(pool);
  setExpensesDb(pool);
  setCompanyDetailsDb(pool);
  setDieselPurchaseDb(pool);
  setPaymentEntryDb(pool);
  setReceiptsDb(pool);
  setBunksDb(pool);
  setBanksDb(pool);
  setDestinationsDb(pool);
  setExpenseTypesDb(pool);
  companyRoutes.setDb(pool);

  // Use routes
  app.use('/api', authRouter);
  app.use('/api/vehicles', vehiclesRouter);
  app.use('/api/customers', customersRouter);
  app.use('/api/drivers', driversRouter);
  app.use('/api/invoices', invoicesRouter);
  app.use('/api/settings', settingsRouter);
  app.use('/api/expenses', expensesRouter);
  app.use('/api/company', companyRoutes.router);
  app.use('/api/company-details', companyDetailsRouter);
  app.use('/api/diesel-purchase', dieselPurchaseRouter);
  app.use('/api/payment-entry', paymentEntryRouter);
  app.use('/api/receipts', receiptsRouter);
  app.use('/api/bunks', bunksRouter);
  app.use('/api/banks', banksRouter);
  app.use('/api/destinations', destinationsRouter);
  app.use('/api/transactions', transactionsRouter);
  app.use('/api/expense-types', expenseTypesRouter);
  app.use('/api/bill-numbers', billNumbersRouter);

  const PORT = process.env.PORT || 5000;
  const server = app.listen(PORT, () => {
    console.log(`Backend server running on port ${PORT}`);
    console.log('Server is ready to accept connections');
  });

  // Keep the server alive
  server.on('error', (error) => {
    console.error('Server error:', error);
  });

  // Prevent the process from exiting
  process.on('SIGINT', () => {
    console.log('Received SIGINT, shutting down gracefully');
    server.close(() => {
      console.log('Server closed');
      process.exit(0);
    });
  });

  // Add error handling for unhandled promise rejections
  process.on('unhandledRejection', (reason, promise) => {
    console.error('Unhandled Rejection at:', promise, 'reason:', reason);
  });

  // Add error handling for uncaught exceptions
  process.on('uncaughtException', (error) => {
    console.error('Uncaught Exception:', error);
    process.exit(1);
  });

  // Add error handling for the server
  app.use((error, req, res, next) => {
    console.error('Server error:', error);
    res.status(500).json({ message: 'Internal server error', error: error.message });
  });
