const mysql = require('mysql2/promise');
const crypto = require('crypto');
require('dotenv').config();

// Set timezone to Indian time
process.env.TZ = 'Asia/Kolkata';

function sha256(str) {
  return crypto.createHash('sha256').update(str).digest('hex');
}

async function setupDatabase() {
  let connection;
  try {
    // Step 1: Connect to MySQL without specifying database
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      timezone: '+05:30', // Indian timezone
    });
    console.log('Connected to MySQL server');
    await connection.execute('CREATE DATABASE IF NOT EXISTS cement');
    console.log('Database "cement" created or already exists');
    await connection.end();

    // Step 2: Connect to the cement database
    connection = await mysql.createConnection({
      host: process.env.DB_HOST || 'localhost',
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'cement',
      timezone: '+05:30', // Indian timezone
    });
    
    // Set timezone for the session
    await connection.execute("SET time_zone = '+05:30'");
    
    // Create admins table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS admins (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "admins" created or already exists');
    
    // Create vehicles table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS vehicles (
        id INT AUTO_INCREMENT PRIMARY KEY,
        number VARCHAR(255) UNIQUE NOT NULL,
        name VARCHAR(255) NOT NULL,
        year INT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "vehicles" created or already exists');
    
    // Create customers table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS customers (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        contact VARCHAR(255) NOT NULL,
        email VARCHAR(255),
        location VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "customers" created or already exists');
    
    // Create drivers table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS drivers (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        phone VARCHAR(255) NOT NULL,
        alt_phone VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "drivers" created or already exists');
    
    // Create companies table for multiple companies
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS companies (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        gst VARCHAR(50) NOT NULL,
        phone VARCHAR(50) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "companies" created or already exists');
    
    // Create invoices table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS invoices (
        id INT AUTO_INCREMENT PRIMARY KEY,
        supplier_invoice_number VARCHAR(255) UNIQUE NOT NULL,
        supplier_invoice_date DATE NOT NULL,
        company_id INT,
        customer_id INT NOT NULL,
        from_location VARCHAR(255) NOT NULL,
        destination VARCHAR(255) NOT NULL,
        quantity DECIMAL(10,2) NOT NULL,
        rate DECIMAL(10,2) NOT NULL,
        amount DECIMAL(10,2) NOT NULL,
        total DECIMAL(10,2) NOT NULL,
        advance_paid DECIMAL(10,2) DEFAULT 0,
        balance DECIMAL(10,2) DEFAULT 0,
        driver_id INT,
        vehicle_id INT,
        driver_charges JSON,
        status VARCHAR(50) DEFAULT 'pending',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        bill_no VARCHAR(255),
        bill_created_at TIMESTAMP NULL,
        paid_date DATE NULL,
        settings_id INT,
        image LONGBLOB,
        FOREIGN KEY (company_id) REFERENCES company_details(id) ON DELETE SET NULL,
        FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE CASCADE,
        FOREIGN KEY (driver_id) REFERENCES drivers(id) ON DELETE SET NULL,
        FOREIGN KEY (vehicle_id) REFERENCES vehicles(id) ON DELETE SET NULL,
        FOREIGN KEY (settings_id) REFERENCES company_details(id) ON DELETE SET NULL
      )
    `);
    console.log('Table "invoices" created or already exists');
    
    // Add new columns if they don't exist (for existing databases)
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN supplier_invoice_number VARCHAR(255) UNIQUE');
      console.log('Added supplier_invoice_number column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN supplier_invoice_date DATE');
      console.log('Added supplier_invoice_date column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN paid_date DATE NULL');
      console.log('Added paid_date column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN settings_id INT');
      console.log('Added settings_id column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE invoices ADD CONSTRAINT fk_invoices_settings FOREIGN KEY (settings_id) REFERENCES company_details(id) ON DELETE SET NULL');
      console.log('Added settings_id foreign key constraint');
    } catch (error) {
      // Constraint already exists
    }
    
    // Remove company invoice columns if they exist (for existing databases)
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN company_invoice_number');
      console.log('Removed company_invoice_number column');
    } catch (error) {
      // Column doesn't exist
    }
    
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN company_invoice_date');
      console.log('Removed company_invoice_date column');
    } catch (error) {
      // Column doesn't exist
    }
    
    // Remove old columns if they exist (for existing databases)
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN invoice_number');
      console.log('Removed invoice_number column');
    } catch (error) {
      // Column doesn't exist
    }
    
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN fuel_charges');
      console.log('Removed fuel_charges column');
    } catch (error) {
      // Column doesn't exist
    }
    
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN toll_charges');
      console.log('Removed toll_charges column');
    } catch (error) {
      // Column doesn't exist
    }
    
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN discount');
      console.log('Removed discount column');
    } catch (error) {
      // Column doesn't exist
    }
    
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN due');
      console.log('Removed due column');
    } catch (error) {
      // Column doesn't exist
    }
    
    // Create settings table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS settings (
        id INT AUTO_INCREMENT PRIMARY KEY,
        company_name VARCHAR(255) NOT NULL,
        address VARCHAR(255),
        city VARCHAR(255),
        phone VARCHAR(50),
        email VARCHAR(255),
        pan VARCHAR(50),
        bank_name VARCHAR(255),
        account_number VARCHAR(100),
        ifsc VARCHAR(50),
        branch VARCHAR(100),
        account_name VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "settings" created or already exists');
    
    // Create company_details table for multiple companies with bank details
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS company_details (
        id INT AUTO_INCREMENT PRIMARY KEY,
        company_name VARCHAR(255) NOT NULL,
        address VARCHAR(255),
        city VARCHAR(255),
        phone VARCHAR(50),
        email VARCHAR(255),
        pan VARCHAR(50),
        gst VARCHAR(50),
        bank_name VARCHAR(255),
        account_number VARCHAR(100),
        ifsc VARCHAR(50),
        branch VARCHAR(100),
        account_name VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "company_details" created or already exists');
    
    // Create receipts table for payment tracking
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS receipts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        company_id INT NULL,
        customer_id INT NULL,
        driver_id INT NULL,
        vehicle_no VARCHAR(255) NULL,
        bunk_id INT NULL,
        bill_no VARCHAR(255),
        payment_date DATE NOT NULL,
        payment_amount DECIMAL(10,2) NOT NULL,
        payment_mode VARCHAR(50),
        reference_number VARCHAR(255),
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (company_id) REFERENCES companies(id) ON DELETE SET NULL,
        FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL,
        FOREIGN KEY (driver_id) REFERENCES drivers(id) ON DELETE SET NULL,
        FOREIGN KEY (bunk_id) REFERENCES bunks(id) ON DELETE SET NULL
      )
    `);
    console.log('Table "receipts" created or already exists');
    
    // Add new columns to receipts table if they don't exist
    try {
      await connection.execute('ALTER TABLE receipts ADD COLUMN customer_id INT NULL');
      console.log('Added customer_id column to receipts table');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE receipts ADD COLUMN driver_id INT NULL');
      console.log('Added driver_id column to receipts table');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE receipts ADD COLUMN vehicle_no VARCHAR(255) NULL');
      console.log('Added vehicle_no column to receipts table');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE receipts ADD COLUMN bunk_id INT NULL');
      console.log('Added bunk_id column to receipts table');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE receipts ADD COLUMN bank_id INT NULL');
      console.log('Added bank_id column to receipts table');
    } catch (error) {
      // Column already exists
    }
    
    // Add foreign key constraints if they don't exist
    try {
      await connection.execute('ALTER TABLE receipts ADD CONSTRAINT fk_receipts_customer FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL');
      console.log('Added foreign key constraint for customer_id');
    } catch (error) {
      // Constraint already exists
    }
    
    try {
      await connection.execute('ALTER TABLE receipts ADD CONSTRAINT fk_receipts_driver FOREIGN KEY (driver_id) REFERENCES drivers(id) ON DELETE SET NULL');
      console.log('Added foreign key constraint for driver_id');
    } catch (error) {
      // Constraint already exists
    }
    
    try {
      await connection.execute('ALTER TABLE receipts ADD CONSTRAINT fk_receipts_bunk FOREIGN KEY (bunk_id) REFERENCES bunks(id) ON DELETE SET NULL');
      console.log('Added foreign key constraint for bunk_id');
    } catch (error) {
      // Constraint already exists
    }
    
    // Create expenses table if it doesn't exist
    await connection.execute(`
      DROP TABLE IF EXISTS expenses
    `);
    console.log('Dropped existing expenses table');
    
    await connection.execute(`
      CREATE TABLE expenses (
        id INT AUTO_INCREMENT PRIMARY KEY,
        vehicle_id INT NOT NULL,
        driver_id INT,
        type ENUM('fuel', 'driver_charge', 'toll') NOT NULL,
        litre DECIMAL(10,2),
        rate DECIMAL(10,2),
        amount DECIMAL(10,2) NOT NULL,
        expense_date DATE NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (vehicle_id) REFERENCES vehicles(id) ON DELETE CASCADE,
        FOREIGN KEY (driver_id) REFERENCES drivers(id) ON DELETE SET NULL
      )
    `);
    console.log('Table "expenses" created with new structure');
    
    // Create bunks table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS bunks (
        id INT AUTO_INCREMENT PRIMARY KEY,
        bunk_name VARCHAR(255) NOT NULL,
        location VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "bunks" created or already exists');
    
    // Create diesel_purchases table if it doesn't exist
    await connection.execute(`
      DROP TABLE IF EXISTS diesel_purchase_entries
    `);
    console.log('Table "diesel_purchase_entries" dropped');
    
    await connection.execute(`
      DROP TABLE IF EXISTS diesel_purchases
    `);
    console.log('Table "diesel_purchases" dropped');
    
    await connection.execute(`
      CREATE TABLE diesel_purchases (
        id INT AUTO_INCREMENT PRIMARY KEY,
        bill_no VARCHAR(255) NOT NULL,
        bill_date DATE NOT NULL,
        vehicle_id INT NOT NULL,
        driver_name VARCHAR(255),
        litres DECIMAL(10,2) NOT NULL,
        rate DECIMAL(10,2) NOT NULL,
        total_amount DECIMAL(10,2) NOT NULL,
        bunk_id INT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (vehicle_id) REFERENCES vehicles(id) ON DELETE CASCADE,
        FOREIGN KEY (bunk_id) REFERENCES bunks(id) ON DELETE SET NULL
      )
    `);
    console.log('Table "diesel_purchases" created with simplified structure');
    
    // Create payment_entries table if it doesn't exist
    await connection.execute(`
      DROP TABLE IF EXISTS payment_entry_drivers
    `);
    console.log('Table "payment_entry_drivers" dropped');
    
    await connection.execute(`
      DROP TABLE IF EXISTS payment_entries
    `);
    console.log('Table "payment_entries" dropped');
    
    await connection.execute(`
      CREATE TABLE payment_entries (
        id INT AUTO_INCREMENT PRIMARY KEY,
        bill_no VARCHAR(255) NOT NULL,
        bill_date DATE NOT NULL,
        account_type_bank_name VARCHAR(255) NOT NULL,
        bank_id INT,
        driver_id INT,
        driver_name VARCHAR(255) NULL,
        vehicle_no VARCHAR(255) NOT NULL,
        bunk_id INT,
        company_id INT,
        customer_id INT,
        selected_expense_id INT,
        expense_amount DECIMAL(10,2) DEFAULT 0,
        total DECIMAL(10,2) NOT NULL DEFAULT 0,
        note TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (bunk_id) REFERENCES bunks(id) ON DELETE SET NULL,
      FOREIGN KEY (bank_id) REFERENCES banks(id) ON DELETE SET NULL,
      FOREIGN KEY (selected_expense_id) REFERENCES expense_types(id) ON DELETE SET NULL,
      FOREIGN KEY (company_id) REFERENCES company_details(id) ON DELETE SET NULL
      )
    `);
    console.log('Table "payment_entries" created with single table structure');
    
    // Create banks table if it doesn't exist
    await connection.execute(`
      CREATE TABLE IF NOT EXISTS banks (
        id INT AUTO_INCREMENT PRIMARY KEY,
        bank_name VARCHAR(255) NOT NULL,
        account_number VARCHAR(100) NOT NULL,
        ifsc_code VARCHAR(50) NOT NULL,
        branch VARCHAR(100) NOT NULL,
        account_name VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    console.log('Table "banks" created or already exists');
    
    const [existingAdmins] = await connection.execute('SELECT * FROM admins WHERE username = ?', ['admin']);
    if (existingAdmins.length === 0) {
      // Create admin user with SHA-256 hashed password
      const hashedPassword = sha256('admin123');
      await connection.execute('INSERT INTO admins (username, password) VALUES (?, ?)', ['admin', hashedPassword]);
      console.log('Admin user created successfully');
      console.log('Username: admin');
      console.log('Password: admin123');
      console.log('SHA-256 Hash:', hashedPassword);
    } else {
      console.log('Admin user already exists');
    }
    
    const [admins] = await connection.execute('SELECT username FROM admins');
    console.log('Current admin users:', admins.map(a => a.username));

    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN image LONGBLOB');
      console.log('Added image (LONGBLOB) column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN pdf LONGBLOB');
      console.log('Added pdf (LONGBLOB) column');
    } catch (error) {
      // Column already exists
    }
    
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN file_type VARCHAR(20)');
      console.log('Added file_type column');
    } catch (error) {
      // Column already exists
    }
    try {
      await connection.execute('ALTER TABLE invoices DROP COLUMN image_url');
      console.log('Dropped image_url column');
    } catch (error) {
      // Column doesn't exist
    }
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN advance_paid DECIMAL(10,2) DEFAULT 0');
      console.log('Added advance_paid column');
    } catch (error) {
      // Column already exists
    }
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN balance DECIMAL(10,2) DEFAULT 0');
      console.log('Added balance column');
    } catch (error) {
      // Column already exists
    }
    try {
      await connection.execute('ALTER TABLE invoices ADD COLUMN invoice_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP');
      console.log('Added invoice_date column');
    } catch (error) {
      // Column already exists
    }
  } catch (error) {
    console.error('Error setting up database:', error);
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

setupDatabase(); 