const mysql = require('mysql2/promise');

// Database configuration - update these values
const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '', // Update with your password
  database: 'cement' // Update with your database name
};

async function syncCompanies() {
  let connection;
  
  try {
    console.log('Connecting to database...');
    connection = await mysql.createConnection(dbConfig);
    console.log('Connected successfully!');

    console.log('\n=== Current State ===');
    
    // Check companies table
    const [companies] = await connection.execute('SELECT * FROM companies ORDER BY id');
    console.log(`\nCompanies table has ${companies.length} records:`);
    companies.forEach(c => {
      console.log(`- ID: ${c.id}, Name: ${c.name}, GST: ${c.gst}`);
    });

    // Check company_details table
    const [companyDetails] = await connection.execute('SELECT * FROM company_details ORDER BY id');
    console.log(`\nCompany_details table has ${companyDetails.length} records:`);
    companyDetails.forEach(c => {
      console.log(`- ID: ${c.id}, Name: ${c.name || 'N/A'}`);
    });

    // Check payment_entries that have company_id
    const [paymentsWithCompany] = await connection.execute(`
      SELECT id, bill_no, company_id 
      FROM payment_entries 
      WHERE company_id IS NOT NULL
      ORDER BY id
    `);
    console.log(`\nPayment entries with company_id: ${paymentsWithCompany.length}`);
    paymentsWithCompany.forEach(p => {
      console.log(`- Payment ID: ${p.id}, Bill: ${p.bill_no}, Company ID: ${p.company_id}`);
    });

    console.log('\n=== Solution Options ===');
    console.log('1. Add missing companies to company_details table');
    console.log('2. Update payment_entries to use valid company_details IDs');
    console.log('3. Remove the foreign key constraint (not recommended)');

    // Option 1: Add missing companies to company_details
    console.log('\n=== Adding Missing Companies to company_details ===');
    for (const company of companies) {
      try {
        const [existing] = await connection.execute('SELECT id FROM company_details WHERE id = ?', [company.id]);
        if (existing.length === 0) {
          await connection.execute(
            'INSERT INTO company_details (id, name) VALUES (?, ?)',
            [company.id, company.name]
          );
          console.log(`✅ Added company ID ${company.id} (${company.name}) to company_details`);
        } else {
          console.log(`ℹ️  Company ID ${company.id} already exists in company_details`);
        }
      } catch (err) {
        console.log(`❌ Error adding company ID ${company.id}: ${err.message}`);
      }
    }

    // Final check
    console.log('\n=== Final State ===');
    const [finalCompanyDetails] = await connection.execute('SELECT * FROM company_details ORDER BY id');
    console.log(`Company_details table now has ${finalCompanyDetails.length} records:`);
    finalCompanyDetails.forEach(c => {
      console.log(`- ID: ${c.id}, Name: ${c.name || 'N/A'}`);
    });

    console.log('\n✅ Company sync completed!');
    console.log('\nNow you should be able to create payment entries with company_id values that exist in company_details table.');

  } catch (error) {
    console.error('❌ Error:', error.message);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed.');
    }
  }
}

// Run the script
syncCompanies();
