const mysql = require('mysql2/promise');

async function testDriverData() {
  let connection;
  
  try {
    connection = await mysql.createConnection({
      host: 'localhost',
      user: 'root',
      password: '',
      database: 'cement_management'
    });

    console.log('🔍 Testing driver data...');

    // Check drivers
    const [drivers] = await connection.execute('SELECT id, name FROM drivers LIMIT 5');
    console.log('\n📋 Drivers in database:');
    drivers.forEach(d => console.log(`  - ID: ${d.id}, Name: ${d.name}`));

    // Check receipts
    const [receipts] = await connection.execute(`
      SELECT id, driver_id, company_id, bill_no, payment_amount, payment_date 
      FROM receipts 
      ORDER BY created_at DESC 
      LIMIT 10
    `);
    console.log('\n📋 Recent receipts:');
    receipts.forEach(r => console.log(`  - ID: ${r.id}, Driver ID: ${r.driver_id}, Company ID: ${r.company_id}, Bill: ${r.bill_no}, Amount: ₹${r.payment_amount}`));

    // Check diesel purchases
    const [dieselPurchases] = await connection.execute(`
      SELECT id, driver_name, bill_no, total_amount, bill_date 
      FROM diesel_purchases 
      ORDER BY created_at DESC 
      LIMIT 10
    `);
    console.log('\n📋 Recent diesel purchases:');
    dieselPurchases.forEach(d => console.log(`  - ID: ${d.id}, Driver: ${d.driver_name || 'NULL'}, Bill: ${d.bill_no}, Amount: ₹${d.total_amount}`));

    // Check if driver_name column exists in diesel_purchases
    const [columns] = await connection.execute(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'diesel_purchases'
      AND COLUMN_NAME = 'driver_name'
    `);
    console.log('\n📋 driver_name column exists:', columns.length > 0);

    // Test specific driver (Kavitha Rani)
    const [kavithaDriver] = await connection.execute(`
      SELECT id, name FROM drivers WHERE name LIKE '%Kavitha%'
    `);
    if (kavithaDriver.length > 0) {
      const driver = kavithaDriver[0];
      console.log(`\n🔍 Testing for driver: ${driver.name} (ID: ${driver.id})`);
      
      // Check receipts for this driver
      const [driverReceipts] = await connection.execute(`
        SELECT id, driver_id, bill_no, payment_amount 
        FROM receipts 
        WHERE driver_id = ?
      `, [driver.id]);
      console.log(`📊 Receipts for ${driver.name}:`, driverReceipts.length);
      driverReceipts.forEach(r => console.log(`  - Receipt ${r.id}: ${r.bill_no}, Amount: ₹${r.payment_amount}`));
      
      // Check diesel purchases for this driver
      const [driverDiesel] = await connection.execute(`
        SELECT id, driver_name, bill_no, total_amount 
        FROM diesel_purchases 
        WHERE LOWER(TRIM(driver_name)) = LOWER(TRIM(?))
      `, [driver.name]);
      console.log(`📊 Diesel purchases for ${driver.name}:`, driverDiesel.length);
      driverDiesel.forEach(d => console.log(`  - Diesel ${d.id}: ${d.bill_no}, Amount: ₹${d.total_amount}`));
    }

  } catch (error) {
    console.error('❌ Error testing driver data:', error);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed');
    }
  }
}

testDriverData();
