const axios = require('axios');

const API_BASE = 'http://localhost:5000/api';

async function testReceiptAPI() {
  console.log('🔍 Testing Receipt API with payment_amount mapping...\n');
  
  try {
    // Test 1: Get all receipts and check payment_amount
    console.log('📊 Test 1: Fetching receipts and checking payment_amount field...');
    const response = await axios.get(`${API_BASE}/receipts`);
    
    console.log(`✅ Found ${response.data.length} receipts`);
    
    if (response.data.length > 0) {
      console.log('\n📊 Sample receipt data:');
      const sampleReceipt = response.data[0];
      console.log({
        id: sampleReceipt.id,
        company_id: sampleReceipt.company_id,
        bill_no: sampleReceipt.bill_no,
        payment_amount: sampleReceipt.payment_amount,
        payment_amount_type: typeof sampleReceipt.payment_amount,
        payment_date: sampleReceipt.payment_date,
        payment_mode: sampleReceipt.payment_mode,
        created_at: sampleReceipt.created_at
      });
      
      // Check if payment_amount exists and has value
      if (sampleReceipt.payment_amount) {
        console.log('✅ payment_amount field exists and has value:', sampleReceipt.payment_amount);
      } else {
        console.log('❌ payment_amount field is missing or empty:', sampleReceipt.payment_amount);
      }
    } else {
      console.log('⚠️ No receipts found in database');
    }
    
    // Test 2: Check companies to see which one has receipts
    console.log('\n📊 Test 2: Checking companies...');
    const companiesResponse = await axios.get(`${API_BASE}/company/all`);
    const companies = Array.isArray(companiesResponse.data) ? companiesResponse.data : companiesResponse.data.companies || [];
    
    console.log(`✅ Found ${companies.length} companies:`);
    companies.forEach(company => {
      console.log(`  - ID: ${company.id}, Name: ${company.name}`);
    });
    
    // Test 3: Check receipts by company
    if (companies.length > 0) {
      console.log('\n📊 Test 3: Checking receipts by company...');
      for (const company of companies) {
        const companyReceipts = response.data.filter(receipt => receipt.company_id == company.id);
        console.log(`  - ${company.name} (ID: ${company.id}): ${companyReceipts.length} receipts`);
        
        if (companyReceipts.length > 0) {
          const totalAmount = companyReceipts.reduce((sum, receipt) => sum + (parseFloat(receipt.payment_amount) || 0), 0);
          console.log(`    Total payment_amount: ₹${totalAmount.toLocaleString()}`);
        }
      }
    }
    
  } catch (error) {
    console.error('❌ Error testing receipt API:', error.message);
    if (error.code === 'ECONNREFUSED') {
      console.log('💡 Make sure backend server is running: npm start');
    }
  }
}

// Run the test
testReceiptAPI().catch(console.error);


