const mysql = require('mysql2/promise');

async function testReceiptCreation() {
  let connection;
  
  try {
    connection = await mysql.createConnection({
      host: 'localhost',
      user: 'root',
      password: '',
      database: 'cement_management'
    });

    console.log('🔍 Testing receipt creation with driver_id...');

    // Check if driver_id column exists
    const [columns] = await connection.execute(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'receipts'
      AND COLUMN_NAME = 'driver_id'
    `);

    if (columns.length === 0) {
      console.log('❌ driver_id column does not exist in receipts table');
      console.log('Adding driver_id column...');
      
      try {
        await connection.execute(`
          ALTER TABLE receipts 
          ADD COLUMN driver_id INT NULL 
          AFTER customer_id
        `);
        console.log('✅ Added driver_id column to receipts table');
      } catch (error) {
        console.log('❌ Error adding column:', error.message);
      }
    } else {
      console.log('✅ driver_id column exists in receipts table');
    }

    // Check current receipts structure
    const [structure] = await connection.execute('DESCRIBE receipts');
    console.log('\n📋 receipts table structure:');
    structure.forEach(col => {
      console.log(`  - ${col.Field}: ${col.Type} ${col.Null === 'YES' ? 'NULL' : 'NOT NULL'}`);
    });

    // Test creating a receipt with driver_id
    const testDriverId = 1; // Use first driver
    const testReceiptData = {
      bill_no: 'TEST-001',
      payment_date: '2024-01-15',
      payment_amount: 1000.00,
      payment_mode: 'Cash',
      reference_number: '',
      notes: 'Test receipt',
      company_id: null,
      customer_id: null,
      driver_id: testDriverId,
      bank_id: null,
      bunk_id: null
    };

    console.log('\n🧪 Testing receipt creation with driver_id...');
    console.log('Test data:', testReceiptData);

    // Insert test receipt
    const [result] = await connection.execute(`
      INSERT INTO receipts (
        bill_no, payment_date, payment_amount, payment_mode, 
        reference_number, notes, company_id, customer_id, 
        driver_id, bank_id, bunk_id
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [
      testReceiptData.bill_no,
      testReceiptData.payment_date,
      testReceiptData.payment_amount,
      testReceiptData.payment_mode,
      testReceiptData.reference_number,
      testReceiptData.notes,
      testReceiptData.company_id,
      testReceiptData.customer_id,
      testReceiptData.driver_id,
      testReceiptData.bank_id,
      testReceiptData.bunk_id
    ]);

    console.log('✅ Test receipt created with ID:', result.insertId);

    // Verify the receipt was created with correct driver_id
    const [createdReceipt] = await connection.execute(`
      SELECT id, driver_id, bill_no, payment_amount 
      FROM receipts 
      WHERE id = ?
    `, [result.insertId]);

    console.log('📋 Created receipt:', createdReceipt[0]);

    // Test the backend route query
    const [routeTest] = await connection.execute(`
      SELECT DISTINCT
        r.id,
        r.company_id,
        r.driver_id,
        r.bill_no,
        DATE_FORMAT(r.payment_date, '%Y-%m-%d') as payment_date_formatted,
        r.payment_amount,
        r.payment_mode,
        r.reference_number,
        r.notes,
        r.bank_id,
        r.created_at,
        c.name as company_name,
        b.bank_name,
        b.account_number,
        b.ifsc_code,
        b.branch,
        b.account_name as bank_account_name,
        'receipt' as transaction_type
      FROM receipts r
      LEFT JOIN companies c ON r.company_id = c.id
      LEFT JOIN banks b ON r.bank_id = b.id
      WHERE r.driver_id = ?
      ORDER BY r.payment_date DESC, r.created_at DESC
    `, [testDriverId]);

    console.log(`\n📊 Backend route query result for driver ${testDriverId}:`, routeTest.length);
    routeTest.forEach(r => console.log(`  - Receipt ${r.id}: ${r.bill_no}, Amount: ₹${r.payment_amount}, Driver ID: ${r.driver_id}`));

    // Clean up test data
    await connection.execute('DELETE FROM receipts WHERE bill_no = ?', [testReceiptData.bill_no]);
    console.log('🧹 Test receipt cleaned up');

  } catch (error) {
    console.error('❌ Error testing receipt creation:', error);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed');
    }
  }
}

testReceiptCreation();
