const mysql = require('mysql2/promise');

// Test configuration
const config = {
  host: 'localhost',
  user: 'root',
  password: '',
  database: 'cement'
};

async function testBillGeneration() {
  let connection;
  
  try {
    console.log('🔗 Connecting to database...');
    connection = await mysql.createConnection(config);
    console.log('✅ Connected to database');

    // Test 1: Check if company details exist
    console.log('\n📋 Test 1: Checking company details...');
    const [companies] = await connection.execute('SELECT * FROM company_details LIMIT 5');
    console.log(`Found ${companies.length} companies in company_details table`);
    if (companies.length > 0) {
      console.log('Sample company:', companies[0]);
    }

    // Test 2: Check if invoices exist with new structure
    console.log('\n📋 Test 2: Checking invoices structure...');
    const [invoices] = await connection.execute('SELECT id, supplier_invoice_number, supplier_invoice_date, company_id, status FROM invoices LIMIT 5');
    console.log(`Found ${invoices.length} invoices`);
    if (invoices.length > 0) {
      console.log('Sample invoice:', invoices[0]);
    }

    // Test 3: Check pending invoices for bill generation
    console.log('\n📋 Test 3: Checking pending invoices...');
    const [pendingInvoices] = await connection.execute(`
      SELECT id, supplier_invoice_number, supplier_invoice_date, company_id, status, total 
      FROM invoices 
      WHERE status = 'pending' OR status IS NULL 
      LIMIT 5
    `);
    console.log(`Found ${pendingInvoices.length} pending invoices`);
    if (pendingInvoices.length > 0) {
      console.log('Sample pending invoice:', pendingInvoices[0]);
    }

    // Test 4: Test bill generation API call
    console.log('\n📋 Test 4: Testing bill generation...');
    if (pendingInvoices.length > 0) {
      const testInvoiceIds = pendingInvoices.slice(0, 2).map(inv => inv.id);
      console.log('Test invoice IDs:', testInvoiceIds);
      
      // This would be the API call structure
      const billData = {
        invoiceIds: testInvoiceIds,
        bill_no: 'TEST-BILL-001',
        bill_created_at: new Date().toISOString().slice(0, 10)
      };
      console.log('Bill generation data:', billData);
      console.log('✅ Bill generation test data prepared');
    }

    // Test 5: Check PDF generation data
    console.log('\n📋 Test 5: Checking PDF generation data...');
    const [billInvoices] = await connection.execute(`
      SELECT 
        i.id,
        i.supplier_invoice_number,
        i.supplier_invoice_date,
        i.customer_id,
        i.destination,
        i.quantity,
        i.rate,
        i.total,
        i.bill_no,
        i.bill_created_at,
        c.name as customer_name
      FROM invoices i
      LEFT JOIN customers c ON i.customer_id = c.id
      WHERE i.bill_no IS NOT NULL
      LIMIT 3
    `);
    console.log(`Found ${billInvoices.length} invoices with bill numbers`);
    if (billInvoices.length > 0) {
      console.log('Sample bill invoice for PDF:', billInvoices[0]);
    }

    console.log('\n✅ All tests completed successfully!');
    console.log('\n📝 Summary:');
    console.log('- Company details table is accessible');
    console.log('- Invoices table has the correct structure (supplier_invoice_number, supplier_invoice_date)');
    console.log('- Pending invoices can be found for bill generation');
    console.log('- Bill generation API structure is ready');
    console.log('- PDF generation data is available');

  } catch (error) {
    console.error('❌ Test failed:', error.message);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\n🔌 Database connection closed');
    }
  }
}

// Run the test
testBillGeneration(); 