const mysql = require('mysql2/promise');

// Database configuration - update these values
const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '', // Update with your password
  database: 'cement_db' // Update with your database name
};

async function updateCompanyIds() {
  let connection;
  
  try {
    console.log('Connecting to database...');
    connection = await mysql.createConnection(dbConfig);
    console.log('Connected successfully!');

    // First, let's see what we have
    console.log('\n=== Current Payment Entries ===');
    const [payments] = await connection.execute(`
      SELECT id, bill_no, driver_name, vehicle_no, company_id 
      FROM payment_entries 
      ORDER BY id
    `);
    
    payments.forEach(p => {
      console.log(`ID: ${p.id}, Bill: ${p.bill_no}, Driver: ${p.driver_name}, Vehicle: ${p.vehicle_no}, Company ID: ${p.company_id}`);
    });

    // Get all companies
    console.log('\n=== Available Companies ===');
    const [companies] = await connection.execute('SELECT id, name FROM companies ORDER BY id');
    companies.forEach(c => {
      console.log(`Company ID: ${c.id}, Name: ${c.name}`);
    });

    // Update payment entries based on driver relationships
    console.log('\n=== Updating Company IDs based on Driver Relationships ===');
    
    for (const payment of payments) {
      if (payment.company_id === null && payment.driver_name && payment.driver_name !== 'General Expense') {
        // Try to find company through driver
        const [driverInvoices] = await connection.execute(`
          SELECT DISTINCT i.company_id, co.name as company_name
          FROM invoices i
          LEFT JOIN companies co ON i.company_id = co.id
          LEFT JOIN drivers d ON i.driver_id = d.id
          WHERE d.name = ? AND i.company_id IS NOT NULL
          LIMIT 1
        `, [payment.driver_name]);

        if (driverInvoices.length > 0) {
          const companyId = driverInvoices[0].company_id;
          const companyName = driverInvoices[0].company_name;
          
          await connection.execute(
            'UPDATE payment_entries SET company_id = ? WHERE id = ?',
            [companyId, payment.id]
          );
          
          console.log(`✅ Updated Payment ID ${payment.id} (${payment.bill_no}) to Company: ${companyName} (ID: ${companyId})`);
        } else {
          console.log(`❌ No company found for Payment ID ${payment.id} (${payment.bill_no}) - Driver: ${payment.driver_name}`);
        }
      }
    }

    // Update payment entries based on vehicle relationships
    console.log('\n=== Updating Company IDs based on Vehicle Relationships ===');
    
    for (const payment of payments) {
      if (payment.company_id === null && payment.vehicle_no && payment.vehicle_no !== 'N/A') {
        // Try to find company through vehicle
        const [vehicleInvoices] = await connection.execute(`
          SELECT DISTINCT i.company_id, co.name as company_name
          FROM invoices i
          LEFT JOIN companies co ON i.company_id = co.id
          LEFT JOIN vehicles v ON i.vehicle_id = v.id
          WHERE v.number = ? AND i.company_id IS NOT NULL
          LIMIT 1
        `, [payment.vehicle_no]);

        if (vehicleInvoices.length > 0) {
          const companyId = vehicleInvoices[0].company_id;
          const companyName = vehicleInvoices[0].company_name;
          
          await connection.execute(
            'UPDATE payment_entries SET company_id = ? WHERE id = ?',
            [companyId, payment.id]
          );
          
          console.log(`✅ Updated Payment ID ${payment.id} (${payment.bill_no}) to Company: ${companyName} (ID: ${companyId})`);
        } else {
          console.log(`❌ No company found for Payment ID ${payment.id} (${payment.bill_no}) - Vehicle: ${payment.vehicle_no}`);
        }
      }
    }

    // Show final results
    console.log('\n=== Final Results ===');
    const [finalPayments] = await connection.execute(`
      SELECT id, bill_no, driver_name, vehicle_no, company_id 
      FROM payment_entries 
      ORDER BY id
    `);
    
    finalPayments.forEach(p => {
      console.log(`ID: ${p.id}, Bill: ${p.bill_no}, Driver: ${p.driver_name}, Vehicle: ${p.vehicle_no}, Company ID: ${p.company_id}`);
    });

    console.log('\n✅ Company ID update process completed!');

  } catch (error) {
    console.error('❌ Error:', error.message);
  } finally {
    if (connection) {
      await connection.end();
      console.log('\nDatabase connection closed.');
    }
  }
}

// Run the script
updateCompanyIds();
