const mysql = require('mysql2/promise');

const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '',
  database: 'cement'
};

/**
 * Check if a bill number already exists across all relevant tables
 * @param {string} billNo - The bill number to check
 * @param {string} excludeTable - Table to exclude from check (for updates)
 * @param {number} excludeId - ID to exclude from check (for updates)
 * @returns {Promise<{isDuplicate: boolean, existingIn: string[]}>}
 */
async function checkDuplicateBillNumber(billNo, excludeTable = null, excludeId = null) {
  let connection;
  
  try {
    connection = await mysql.createConnection(dbConfig);
    
    const tables = [
      { name: 'payment_entries', column: 'bill_no' },
      { name: 'receipts', column: 'bill_no' },
      { name: 'diesel_purchase', column: 'bill_no' },
      { name: 'invoices', column: 'supplier_invoice_number' },
      { name: 'invoices', column: 'bill_no' }
    ];
    
    const existingIn = [];
    
    for (const table of tables) {
      let query = `SELECT id FROM ${table.name} WHERE ${table.column} = ?`;
      let params = [billNo];
      
      // Exclude current record if updating
      if (excludeTable === table.name && excludeId) {
        query += ' AND id != ?';
        params.push(excludeId);
      }
      
      const [rows] = await connection.execute(query, params);
      
      if (rows.length > 0) {
        existingIn.push(`${table.name}.${table.column}`);
      }
    }
    
    return {
      isDuplicate: existingIn.length > 0,
      existingIn: existingIn
    };
    
  } catch (error) {
    console.error('Error checking duplicate bill number:', error);
    throw error;
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

/**
 * Get the next available bill number
 * @returns {Promise<string>}
 */
async function getNextBillNumber() {
  let connection;
  
  try {
    connection = await mysql.createConnection(dbConfig);
    
    const tables = [
      'payment_entries',
      'receipts', 
      'diesel_purchase'
    ];
    
    let maxNumber = 0;
    
    for (const table of tables) {
      const [rows] = await connection.execute(`SELECT bill_no FROM ${table} WHERE bill_no REGEXP '^[0-9]+$'`);
      
      for (const row of rows) {
        const num = parseInt(row.bill_no, 10);
        if (!isNaN(num) && num > maxNumber) {
          maxNumber = num;
        }
      }
    }
    
    return String(maxNumber + 1);
    
  } catch (error) {
    console.error('Error getting next bill number:', error);
    return '1';
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

module.exports = {
  checkDuplicateBillNumber,
  getNextBillNumber
};





