const mysql = require('mysql2/promise');

const dbConfig = {
  host: 'localhost',
  user: 'root',
  password: '',
  database: 'cement'
};

/**
 * Get the next sequential bill number for a specific table
 * @param {string} tableName - The table name to check
 * @param {string} columnName - The column name (usually 'bill_no')
 * @returns {Promise<string>}
 */
async function getNextSequentialBillNumber(tableName, columnName = 'bill_no') {
  let connection;
  
  try {
    connection = await mysql.createConnection(dbConfig);
    
    // Get all existing bill numbers as integers
    const [rows] = await connection.execute(
      `SELECT ${columnName} FROM ${tableName} WHERE ${columnName} REGEXP '^[0-9]+$' ORDER BY CAST(${columnName} AS UNSIGNED)`
    );
    
    const existingNumbers = rows.map(row => parseInt(row[columnName], 10)).filter(num => !isNaN(num));
    
    // If no numbers exist, start with 1
    if (existingNumbers.length === 0) {
      return '1';
    }
    
    // Find the first gap in the sequence, starting from 1
    let nextNumber = 1;
    for (const existingNum of existingNumbers) {
      if (existingNum === nextNumber) {
        nextNumber++;
      } else {
        // Found a gap, return this number
        return String(nextNumber);
      }
    }
    
    // No gaps found, return the next number after the highest
    return String(nextNumber);
    
  } catch (error) {
    console.error(`Error getting next sequential bill number for ${tableName}:`, error);
    return '1';
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

/**
 * Get the next sequential bill number across all relevant tables
 * This ensures no duplicates across payment_entries, receipts, diesel_purchase
 * @returns {Promise<string>}
 */
async function getNextGlobalSequentialBillNumber() {
  let connection;
  
  try {
    connection = await mysql.createConnection(dbConfig);
    
    const tables = [
      { name: 'payment_entries', column: 'bill_no' },
      { name: 'receipts', column: 'bill_no' },
      { name: 'diesel_purchase', column: 'bill_no' }
    ];
    
    const allNumbers = new Set();
    
    // Collect all bill numbers from all tables
    for (const table of tables) {
      try {
        const [rows] = await connection.execute(
          `SELECT ${table.column} FROM ${table.name} WHERE ${table.column} REGEXP '^[0-9]+$'`
        );
        
        rows.forEach(row => {
          const num = parseInt(row[table.column], 10);
          if (!isNaN(num)) {
            allNumbers.add(num);
          }
        });
      } catch (error) {
        console.error(`Error fetching from ${table.name}:`, error);
        // Continue with other tables
      }
    }
    
    // Convert to sorted array
    const sortedNumbers = Array.from(allNumbers).sort((a, b) => a - b);
    
    // If no numbers exist, start with 1
    if (sortedNumbers.length === 0) {
      return '1';
    }
    
    // Find the first gap in the sequence, starting from 1
    let nextNumber = 1;
    for (const existingNum of sortedNumbers) {
      if (existingNum === nextNumber) {
        nextNumber++;
      } else {
        // Found a gap, return this number
        return String(nextNumber);
      }
    }
    
    // No gaps found, return the next number after the highest
    return String(nextNumber);
    
  } catch (error) {
    console.error('Error getting next global sequential bill number:', error);
    return '1';
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

/**
 * Check if a bill number already exists across all tables
 * @param {string} billNo - The bill number to check
 * @returns {Promise<boolean>}
 */
async function isBillNumberExists(billNo) {
  let connection;
  
  try {
    connection = await mysql.createConnection(dbConfig);
    
    const tables = [
      { name: 'payment_entries', column: 'bill_no' },
      { name: 'receipts', column: 'bill_no' },
      { name: 'diesel_purchase', column: 'bill_no' },
      { name: 'invoices', column: 'supplier_invoice_number' },
      { name: 'invoices', column: 'bill_no' }
    ];
    
    for (const table of tables) {
      try {
        const [rows] = await connection.execute(
          `SELECT id FROM ${table.name} WHERE ${table.column} = ?`,
          [billNo]
        );
        
        if (rows.length > 0) {
          return true;
        }
      } catch (error) {
        console.error(`Error checking ${table.name}:`, error);
        // Continue with other tables
      }
    }
    
    return false;
    
  } catch (error) {
    console.error('Error checking bill number existence:', error);
    return false;
  } finally {
    if (connection) {
      await connection.end();
    }
  }
}

module.exports = {
  getNextSequentialBillNumber,
  getNextGlobalSequentialBillNumber,
  isBillNumberExists
};





