const express = require('express');
const router = express.Router();

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// Get all banks
router.get('/', async (req, res) => {
    try {
        const [rows] = await db.query('SELECT * FROM banks ORDER BY id ASC');
        res.json(rows);
    } catch (error) {
        console.error('Error fetching banks:', error);
        res.status(500).json({ message: 'Error fetching banks' });
    }
});

// Add a new bank
router.post('/', async (req, res) => {
    const { bank_name, account_number, ifsc_code, branch, account_name } = req.body;
    if (!bank_name || !account_number || !ifsc_code || !branch || !account_name) {
        return res.status(400).json({ message: 'All fields are required' });
    }

    try {
        const [result] = await db.query(
            'INSERT INTO banks (bank_name, account_number, ifsc_code, branch, account_name) VALUES (?, ?, ?, ?, ?)',
            [bank_name, account_number, ifsc_code, branch, account_name]
        );
        res.status(201).json({ id: result.insertId, bank_name, account_number, ifsc_code, branch, account_name });
    } catch (error) {
        console.error('Error adding bank:', error);
        res.status(500).json({ message: 'Error adding bank' });
    }
});

// Get a single bank by ID
router.get('/:id', async (req, res) => {
    const { id } = req.params;
    try {
        const [rows] = await db.query('SELECT * FROM banks WHERE id = ?', [id]);
        if (rows.length === 0) {
            return res.status(404).json({ message: 'Bank not found' });
        }
        res.json(rows[0]);
    } catch (error) {
        console.error('Error fetching bank:', error);
        res.status(500).json({ message: 'Error fetching bank' });
    }
});

// Update a bank
router.put('/:id', async (req, res) => {
    const { id } = req.params;
    const { bank_name, account_number, ifsc_code, branch, account_name } = req.body;
    
    if (!bank_name || !account_number || !ifsc_code || !branch || !account_name) {
        return res.status(400).json({ message: 'All fields are required' });
    }

    try {
        // Check if bank exists
        const [existing] = await db.query('SELECT id FROM banks WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({ message: 'Bank not found' });
        }

        const [result] = await db.query(
            'UPDATE banks SET bank_name = ?, account_number = ?, ifsc_code = ?, branch = ?, account_name = ? WHERE id = ?',
            [bank_name, account_number, ifsc_code, branch, account_name, id]
        );
        
        res.json({ message: 'Bank updated successfully' });
    } catch (error) {
        console.error('Error updating bank:', error);
        res.status(500).json({ message: 'Error updating bank' });
    }
});

// Delete a bank
router.delete('/:id', async (req, res) => {
    const { id } = req.params;
    try {
        // Check if bank exists
        const [existing] = await db.query('SELECT id FROM banks WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({ message: 'Bank not found' });
        }

        await db.query('DELETE FROM banks WHERE id = ?', [id]);
        res.json({ message: 'Bank deleted successfully' });
    } catch (error) {
        console.error('Error deleting bank:', error);
        res.status(500).json({ message: 'Error deleting bank' });
    }
});

module.exports = { router, setDb }; 