const express = require('express');
const router = express.Router();

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// Get all bunks
router.get('/', async (req, res) => {
    try {
        const [rows] = await db.query('SELECT * FROM bunks ORDER BY created_at ASC');
        res.json(rows);
    } catch (error) {
        console.error('Error fetching bunks:', error);
        res.status(500).json({ message: 'Error fetching bunks' });
    }
});

// Add a new bunk
router.post('/', async (req, res) => {
    const { bunk_name, location } = req.body;
    if (!bunk_name) {
        return res.status(400).json({ message: 'Bunk name is required' });
    }

    try {
        const [result] = await db.query(
            'INSERT INTO bunks (bunk_name, location) VALUES (?, ?)',
            [bunk_name, location]
        );
        res.status(201).json({ id: result.insertId, bunk_name, location });
    } catch (error) {
        console.error('Error adding bunk:', error);
        res.status(500).json({ message: 'Error adding bunk' });
    }
});

// Update a bunk
router.put('/:id', async (req, res) => {
    const { id } = req.params;
    const { bunk_name, location } = req.body;

    console.log(`Received PUT request for bunk. ID: ${id}. Body:`, req.body);

    if (!bunk_name) {
        return res.status(400).json({ message: 'Bunk name is required' });
    }

    try {
        const [result] = await db.query(
            'UPDATE bunks SET bunk_name = ?, location = ? WHERE id = ?',
            [bunk_name, location, id]
        );

        if (result.affectedRows === 0) {
            return res.status(404).json({ message: 'Bunk not found' });
        }

        res.json({ message: 'Bunk updated successfully' });
    } catch (error) {
        console.error('Error updating bunk:', error);
        res.status(500).json({ message: 'Error updating bunk' });
    }
});

// Delete a bunk
router.delete('/:id', async (req, res) => {
    const { id } = req.params;
    try {
        const [result] = await db.query('DELETE FROM bunks WHERE id = ?', [id]);
        if (result.affectedRows === 0) {
            return res.status(404).json({ message: 'Bunk not found' });
        }
        res.json({ message: 'Bunk deleted successfully' });
    } catch (error) {
        console.error('Error deleting bunk:', error);
        // Check if it's a foreign key constraint error
        if (error.code === 'ER_ROW_IS_REFERENCED_2' || error.errno === 1451) {
            return res.status(400).json({ 
                message: 'Cannot delete this bunk because it is being used in diesel purchases. Please delete the related diesel purchases first.',
                error: 'foreign key constraint'
            });
        }
        res.status(500).json({ message: 'Error deleting bunk', error: error.message });
    }
});

module.exports = { router, setDb }; 