const express = require('express');
const router = express.Router();

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// Get all cash transactions (aggregated from multiple sources)
router.get('/', async (req, res) => {
    try {
        console.log('🔍 Fetching cash transactions...');
        
        // Get cash transactions from cash_transactions table
        const [cashTransactions] = await db.query(`
            SELECT 
                c.id,
                c.bill_no,
                c.transaction_date as date,
                c.amount,
                b.bank_name,
                b.account_number,
                c.mode,
                c.description,
                c.reference_number,
                'Cash Transaction' as source,
                c.created_at
            FROM cash_transactions c
            LEFT JOIN banks b ON c.bank_id = b.id
        `);
        console.log('💰 Cash transactions found:', cashTransactions.length);

        // Get payment entries (ONLY cash mode)
        const [paymentEntries] = await db.query(`
            SELECT 
                pe.id,
                pe.bill_no,
                pe.bill_date as date,
                pe.total as amount,
                pe.account_type_bank_name as bank_name,
                NULL as account_number,
                'cash' as mode,
                CONCAT(
                    COALESCE(pe.driver_name, ''), 
                    ' - ', 
                    COALESCE(pe.vehicle_no, ''),
                    CASE 
                        WHEN b.bunk_name IS NOT NULL THEN CONCAT(' - Bunk: ', b.bunk_name)
                        ELSE ''
                    END
                ) as description,
                NULL as reference_number,
                'Payment Entry' as source,
                pe.created_at
            FROM payment_entries pe
            LEFT JOIN bunks b ON pe.bunk_id = b.id
            WHERE LOWER(pe.account_type_bank_name) = 'cash'
        `);
        console.log('💳 Payment entries found:', paymentEntries.length);

        // Get receipts (ONLY cash mode)
        const [receipts] = await db.query(`
            SELECT 
                r.id,
                r.bill_no,
                r.payment_date as date,
                r.payment_amount as amount,
                b.bank_name,
                b.account_number,
                COALESCE(r.payment_mode, 'cash') as mode,
                CONCAT(
                    'Receipt from ',
                    COALESCE(c.name, 'Company'),
                    CASE 
                        WHEN r.notes IS NOT NULL THEN CONCAT(' - ', r.notes)
                        ELSE ''
                    END
                ) as description,
                r.reference_number as reference_number,
                'Receipt' as source,
                r.created_at
            FROM receipts r
            LEFT JOIN banks b ON r.bank_id = b.id
            LEFT JOIN companies c ON r.company_id = c.id
            WHERE COALESCE(r.payment_mode, 'cash') = 'cash'
        `);
        console.log('🧾 Receipts found:', receipts.length);

        // Get diesel purchases (all modes) - diesel_purchases doesn't have payment_mode column
        const [dieselPurchases] = await db.query(`
            SELECT 
                dp.id,
                dp.bill_no,
                dp.bill_date as date,
                dp.total_amount as amount,
                bk.bunk_name as bank_name,
                NULL as account_number,
                'cash' as mode,
                CONCAT(
                    'Diesel: ', 
                    COALESCE(dp.litres, 0),
                    'L',
                    CASE 
                        WHEN dp.driver_name IS NOT NULL THEN CONCAT(' - Driver: ', dp.driver_name)
                        ELSE ''
                    END,
                    CASE 
                        WHEN v.number IS NOT NULL THEN CONCAT(' - Vehicle: ', v.number)
                        ELSE ''
                    END
                ) as description,
                NULL as reference_number,
                'Diesel Purchase' as source,
                dp.created_at
            FROM diesel_purchases dp
            LEFT JOIN bunks bk ON dp.bunk_id = bk.id
            LEFT JOIN vehicles v ON dp.vehicle_id = v.id
        `);
        console.log('⛽ Diesel purchases found:', dieselPurchases.length);

        // Combine all transactions
        const allTransactions = [
            ...cashTransactions,
            ...paymentEntries,
            ...receipts,
            ...dieselPurchases
        ];

        console.log('📊 Total transactions found:', allTransactions.length);

        // Sort by date (newest first)
        allTransactions.sort((a, b) => {
            const dateA = new Date(a.date);
            const dateB = new Date(b.date);
            return dateB - dateA;
        });

        console.log('✅ Sending transactions to frontend');
        res.json(allTransactions);
    } catch (error) {
        console.error('Error fetching cash transactions:', error);
        res.status(500).json({ message: 'Error fetching cash transactions', error: error.message });
    }
});

// Get a single cash transaction by ID
router.get('/:id', async (req, res) => {
    const { id } = req.params;
    try {
        const [rows] = await db.query(`
            SELECT 
                c.*,
                b.bank_name,
                b.account_number
            FROM cash_transactions c
            LEFT JOIN banks b ON c.bank_id = b.id
            WHERE c.id = ?
        `, [id]);
        
        if (rows.length === 0) {
            return res.status(404).json({ message: 'Cash transaction not found' });
        }
        res.json(rows[0]);
    } catch (error) {
        console.error('Error fetching cash transaction:', error);
        res.status(500).json({ message: 'Error fetching cash transaction', error: error.message });
    }
});

// Add a new cash transaction
router.post('/', async (req, res) => {
    const { bill_no, transaction_date, amount, bank_id, mode, description, reference_number } = req.body;
    
    // Validation
    if (!bill_no || !transaction_date || !amount || !mode) {
        return res.status(400).json({ message: 'Bill number, date, amount, and mode are required' });
    }

    try {
        // Check if bill_no already exists
        const [existing] = await db.query('SELECT id FROM cash_transactions WHERE bill_no = ?', [bill_no]);
        if (existing.length > 0) {
            return res.status(400).json({ message: 'Bill number already exists' });
        }

        const [result] = await db.query(
            `INSERT INTO cash_transactions 
            (bill_no, transaction_date, amount, bank_id, mode, description, reference_number) 
            VALUES (?, ?, ?, ?, ?, ?, ?)`,
            [bill_no, transaction_date, amount, bank_id || null, mode, description || null, reference_number || null]
        );
        
        // Fetch the newly created transaction with bank details
        const [newTransaction] = await db.query(`
            SELECT 
                c.*,
                b.bank_name,
                b.account_number
            FROM cash_transactions c
            LEFT JOIN banks b ON c.bank_id = b.id
            WHERE c.id = ?
        `, [result.insertId]);

        res.status(201).json(newTransaction[0]);
    } catch (error) {
        console.error('Error adding cash transaction:', error);
        res.status(500).json({ message: 'Error adding cash transaction', error: error.message });
    }
});

// Update a cash transaction
router.put('/:id', async (req, res) => {
    const { id } = req.params;
    const { bill_no, transaction_date, amount, bank_id, mode, description, reference_number } = req.body;
    
    if (!bill_no || !transaction_date || !amount || !mode) {
        return res.status(400).json({ message: 'Bill number, date, amount, and mode are required' });
    }

    try {
        // Check if cash transaction exists
        const [existing] = await db.query('SELECT id FROM cash_transactions WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({ message: 'Cash transaction not found' });
        }

        // Check if bill_no is being changed and if new bill_no already exists
        const [billCheck] = await db.query('SELECT id FROM cash_transactions WHERE bill_no = ? AND id != ?', [bill_no, id]);
        if (billCheck.length > 0) {
            return res.status(400).json({ message: 'Bill number already exists' });
        }

        await db.query(
            `UPDATE cash_transactions 
            SET bill_no = ?, transaction_date = ?, amount = ?, bank_id = ?, mode = ?, 
                description = ?, reference_number = ?, updated_at = NOW()
            WHERE id = ?`,
            [bill_no, transaction_date, amount, bank_id || null, mode, description || null, reference_number || null, id]
        );
        
        // Fetch the updated transaction with bank details
        const [updatedTransaction] = await db.query(`
            SELECT 
                c.*,
                b.bank_name,
                b.account_number
            FROM cash_transactions c
            LEFT JOIN banks b ON c.bank_id = b.id
            WHERE c.id = ?
        `, [id]);

        res.json(updatedTransaction[0]);
    } catch (error) {
        console.error('Error updating cash transaction:', error);
        res.status(500).json({ message: 'Error updating cash transaction', error: error.message });
    }
});

// Delete a cash transaction
router.delete('/:id', async (req, res) => {
    const { id } = req.params;
    try {
        // Check if cash transaction exists
        const [existing] = await db.query('SELECT id FROM cash_transactions WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({ message: 'Cash transaction not found' });
        }

        await db.query('DELETE FROM cash_transactions WHERE id = ?', [id]);
        res.json({ message: 'Cash transaction deleted successfully' });
    } catch (error) {
        console.error('Error deleting cash transaction:', error);
        res.status(500).json({ message: 'Error deleting cash transaction', error: error.message });
    }
});

// Get cash summary (from all sources)
router.get('/summary/total', async (req, res) => {
    try {
        // Count and sum from each table
        const [cashTrans] = await db.query(`
            SELECT COUNT(*) as count, COALESCE(SUM(amount), 0) as total FROM cash_transactions
        `);
        
        const [payments] = await db.query(`
            SELECT COUNT(*) as count, COALESCE(SUM(total), 0) as total FROM payment_entries
            WHERE LOWER(account_type_bank_name) = 'cash'
        `);
        
        const [receipts] = await db.query(`
            SELECT COUNT(*) as count, COALESCE(SUM(payment_amount), 0) as total FROM receipts
            WHERE LOWER(COALESCE(payment_mode, 'cash')) = 'cash'
        `);
        
        const [diesel] = await db.query(`
            SELECT COUNT(*) as count, COALESCE(SUM(total_amount), 0) as total FROM diesel_purchases
        `);

        // Sum by mode from all tables
        // Note: payment_entries and diesel_purchases don't have payment_mode column, so we default to 'cash'
        const [modeBreakdown] = await db.query(`
            SELECT 
                SUM(CASE WHEN mode = 'cash' THEN amount ELSE 0 END) as cash_total,
                SUM(CASE WHEN mode = 'bank' THEN amount ELSE 0 END) as bank_total,
                SUM(CASE WHEN mode = 'upi' THEN amount ELSE 0 END) as upi_total,
                SUM(CASE WHEN mode = 'cheque' THEN amount ELSE 0 END) as cheque_total
            FROM (
                SELECT amount, CAST(mode AS CHAR) as mode FROM cash_transactions
                UNION ALL
                SELECT total as amount, CAST('cash' AS CHAR) as mode FROM payment_entries
                WHERE LOWER(account_type_bank_name) = 'cash'
                UNION ALL
                SELECT payment_amount as amount, CAST(COALESCE(payment_mode, 'cash') AS CHAR) as mode FROM receipts
                UNION ALL
                SELECT total_amount as amount, CAST('cash' AS CHAR) as mode FROM diesel_purchases
            ) all_transactions
        `);

        const summary = {
            total_transactions: parseInt(cashTrans[0].count) + parseInt(payments[0].count) + 
                               parseInt(receipts[0].count) + parseInt(diesel[0].count),
            total_amount: parseFloat(cashTrans[0].total) + parseFloat(payments[0].total) + 
                         parseFloat(receipts[0].total) + parseFloat(diesel[0].total),
            cash_total: parseFloat(modeBreakdown[0].cash_total || 0),
            bank_total: parseFloat(modeBreakdown[0].bank_total || 0),
            upi_total: parseFloat(modeBreakdown[0].upi_total || 0),
            cheque_total: parseFloat(modeBreakdown[0].cheque_total || 0),
            payment_entries: parseInt(payments[0].count),
            receipts: parseInt(receipts[0].count),
            diesel_purchases: parseInt(diesel[0].count),
            cash_transactions: parseInt(cashTrans[0].count)
        };

        res.json(summary);
    } catch (error) {
        console.error('Error fetching cash summary:', error);
        res.status(500).json({ message: 'Error fetching cash summary', error: error.message });
    }
});

module.exports = { router, setDb };

