const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// Get all company details
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT * FROM company_details 
      ORDER BY created_at ASC
    `);
    
    // Convert timestamps to Indian time
    const companies = rows.map(company => {
      const createdAt = new Date(company.created_at);
      const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
      
      // Format created_at to show only date (DD/MM/YYYY)
      let formattedCreatedAt = '';
      if (company.created_at) {
        const createdDate = new Date(company.created_at);
        const day = createdDate.getDate().toString().padStart(2, '0');
        const month = (createdDate.getMonth() + 1).toString().padStart(2, '0');
        const year = createdDate.getFullYear();
        formattedCreatedAt = `${day}/${month}/${year}`;
      }
      
      return {
        ...company,
        created_at: formattedCreatedAt
      };
    });
    
    res.json(companies);
  } catch (err) {
    console.error('Error fetching company details:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get single company detail by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    const [rows] = await db.query('SELECT * FROM company_details WHERE id = ?', [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Company detail not found' });
    }
    
    const company = rows[0];
    
    // Convert timestamps to Indian time
    const createdAt = new Date(company.created_at);
    const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
    
    company.created_at = indianCreatedAt.toISOString();
    
    res.json(company);
  } catch (err) {
    console.error('Error fetching company detail:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Create new company detail
router.post('/', async (req, res) => {
  const {
    company_name,
    address,
    city,
    phone,
    email,
    pan,
    gst,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name
  } = req.body;
  
  // Validation
  if (!company_name) {
    return res.status(400).json({ 
      message: 'Company name is required' 
    });
  }

  try {
    const [result] = await db.query(`
      INSERT INTO company_details (
        company_name, address, city, phone, email, pan, gst, 
        bank_name, account_number, ifsc, branch, account_name
      )
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [company_name, address, city, phone, email, pan, gst, bank_name, account_number, ifsc, branch, account_name]);
    
    res.status(201).json({ 
      message: 'Company detail created successfully',
      id: result.insertId
    });
  } catch (err) {
    console.error('Error creating company detail:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update company detail
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const {
    company_name,
    address,
    city,
    phone,
    email,
    pan,
    gst,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name
  } = req.body;
  
  // Validation
  if (!company_name) {
    return res.status(400).json({ 
      message: 'Company name is required' 
    });
  }

  try {
    // Check if company detail exists
    const [existing] = await db.query('SELECT id FROM company_details WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Company detail not found' });
    }

    await db.query(`
      UPDATE company_details SET 
        company_name = ?, address = ?, city = ?, phone = ?, email = ?, 
        pan = ?, gst = ?, bank_name = ?, account_number = ?, ifsc = ?, 
        branch = ?, account_name = ?
      WHERE id = ?
    `, [company_name, address, city, phone, email, pan, gst, bank_name, account_number, ifsc, branch, account_name, id]);
    
    res.json({ message: 'Company detail updated successfully' });
  } catch (err) {
    console.error('Error updating company detail:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete company detail
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    // Check if company detail exists
    const [existing] = await db.query('SELECT id FROM company_details WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Company detail not found' });
    }

    await db.query('DELETE FROM company_details WHERE id = ?', [id]);
    res.json({ message: 'Company detail deleted successfully' });
  } catch (err) {
    console.error('Error deleting company detail:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 