const express = require('express');
const router = express.Router();
let db;
function setDb(pool) { db = pool; }

// Get settings (single row)
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT * FROM settings LIMIT 1');
    if (rows.length === 0) return res.json(null);
    res.json(rows[0]);
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add or update settings (upsert)
router.post('/', async (req, res) => {
  const {
    company_name,
    address,
    city,
    phone,
    pan,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name
  } = req.body;
  try {
    // Upsert: if exists, update; else, insert
    const [rows] = await db.query('SELECT id FROM settings LIMIT 1');
    if (rows.length === 0) {
      await db.query(
        'INSERT INTO settings (company_name, address, city, phone, pan, bank_name, account_number, ifsc, branch, account_name) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
        [company_name, address, city, phone, pan, bank_name, account_number, ifsc, branch, account_name]
      );
    } else {
      await db.query(
        'UPDATE settings SET company_name=?, address=?, city=?, phone=?, pan=?, bank_name=?, account_number=?, ifsc=?, branch=?, account_name=? WHERE id=?',
        [company_name, address, city, phone, pan, bank_name, account_number, ifsc, branch, account_name, rows[0].id]
      );
    }
    res.json({ message: 'Settings saved' });
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// --- New company table for multiple companies ---
// GET all companies
router.get('/all', async (req, res) => {
  try {
    console.log('Fetching companies...');
    const { search, page = 1, limit = 10 } = req.query;
    const offset = (parseInt(page) - 1) * parseInt(limit);
    
    // Use company_details table
    const [companyCount] = await db.query('SELECT COUNT(*) as count FROM company_details');
    console.log('Company details count:', companyCount[0].count);
    
    let query = `
      SELECT 
        id,
        company_name as name,
        gst,
        phone,
        created_at,
        address,
        city,
        email,
        pan,
        bank_name,
        account_number,
        ifsc,
        branch,
        account_name
      FROM company_details
    `;
    
    const params = [];
    
    if (search) {
      query += ` WHERE 
        company_name LIKE ? OR 
        gst LIKE ? OR 
        phone LIKE ? OR
        address LIKE ? OR
        city LIKE ? OR
        email LIKE ? OR
        pan LIKE ?
      `;
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm, searchTerm, searchTerm, searchTerm, searchTerm);
    }
    
    // Get total count for pagination
    let countQuery = `
      SELECT COUNT(*) as total
      FROM company_details
    `;
    
    if (search) {
      countQuery += ` WHERE 
        company_name LIKE ? OR 
        gst LIKE ? OR 
        phone LIKE ? OR
        address LIKE ? OR
        city LIKE ? OR
        email LIKE ? OR
        pan LIKE ?
      `;
    }
    
    const [countResult] = await db.query(countQuery, search ? params : []);
    const totalItems = countResult[0].total;
    
    // Add pagination to main query
    query += ` ORDER BY id ASC LIMIT ? OFFSET ?`;
    params.push(parseInt(limit), offset);
    
    const [rows] = await db.query(query, params);
    
    // Return with pagination info
    res.json({
      companies: rows,
      pagination: {
        currentPage: parseInt(page),
        totalPages: Math.ceil(totalItems / parseInt(limit)),
        totalItems: totalItems,
        itemsPerPage: parseInt(limit),
        hasNextPage: parseInt(page) < Math.ceil(totalItems / parseInt(limit)),
        hasPrevPage: parseInt(page) > 1
      }
    });
  } catch (err) {
    console.error('Error fetching companies:', err);
    console.error('Error details:', err.message);
    console.error('Error code:', err.code);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add a new company
router.post('/add', async (req, res) => {
  const { name, gst, phone } = req.body;
  if (!name) return res.status(400).json({ message: 'Company name is required' });
  try {
    const [result] = await db.query(
      'INSERT INTO company_details (company_name, gst, phone) VALUES (?, ?, ?)', 
      [name, gst || '', phone || '']
    );
    res.json({ 
      message: 'Company added',
      id: result.insertId
    });
  } catch (err) {
    console.error('Error adding company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update a company
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const { name, gst, phone } = req.body;
  console.log('Update company request:', { id, name, gst, phone });
  if (!name) return res.status(400).json({ message: 'Company name is required' });
  try {
    const [result] = await db.query('UPDATE company_details SET company_name=?, gst=?, phone=? WHERE id=?', [name, gst || '', phone || '', id]);
    console.log('Update result:', result);
    if (result.affectedRows === 0) return res.status(404).json({ message: 'Company not found' });
    res.json({ message: 'Company updated' });
  } catch (err) {
    console.error('Error updating company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update company details by company ID
router.put('/:id/details', async (req, res) => {
  const { id } = req.params;
  const {
    company_name,
    address,
    city,
    email,
    pan,
    gst,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name: accountName
  } = req.body;
  
  try {
    // First get the company name from the company_details table
    const [companyRows] = await db.query('SELECT company_name FROM company_details WHERE id = ?', [id]);
    if (companyRows.length === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }
    
    const companyName = companyRows[0].company_name;
    
    // Check if company details already exist
    const [existing] = await db.query('SELECT id FROM company_details WHERE company_name = ?', [companyName]);
    
    if (existing.length > 0) {
      // Update existing details
      await db.query(`
        UPDATE company_details 
        SET address = ?, city = ?, email = ?, pan = ?, gst = ?, bank_name = ?, 
            account_number = ?, ifsc = ?, branch = ?, account_name = ?
        WHERE company_name = ?
      `, [address, city, email, pan, gst, bank_name, account_number, ifsc, branch, accountName, companyName]);
    } else {
      // Insert new details
      await db.query(`
        INSERT INTO company_details 
        (company_name, address, city, email, pan, gst, bank_name, account_number, ifsc, branch, account_name)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `, [companyName, address, city, email, pan, gst, bank_name, account_number, ifsc, branch, accountName]);
    }
    
    res.json({ message: 'Company details updated successfully' });
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete a company
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  console.log('Delete company request:', { id });
  try {
    const [result] = await db.query('DELETE FROM company_details WHERE id=?', [id]);
    console.log('Delete result:', result);
    if (result.affectedRows === 0) return res.status(404).json({ message: 'Company not found' });
    res.json({ message: 'Company deleted' });
  } catch (err) {
    console.error('Error deleting company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get a single company by id
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        id,
        company_name as name,
        gst,
        phone,
        created_at,
        address,
        city,
        email,
        pan,
        bank_name,
        account_number,
        ifsc,
        branch,
        account_name
      FROM company_details
      WHERE id = ?
    `, [id]);
    if (rows.length === 0) return res.status(404).json({ message: 'Company not found' });
    res.json(rows[0]);
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Debug route to check company_details table
router.get('/debug/details', async (req, res) => {
  try {
    const [companyDetails] = await db.query('SELECT * FROM company_details');
    const [companies] = await db.query('SELECT * FROM companies');
    
    // Test the JOIN to see what's happening
    const [joinTest] = await db.query(`
      SELECT 
        cd.id,
        cd.company_name,
        cd.gst,
        cd.phone,
        cd.address,
        cd.city
      FROM company_details cd
      ORDER BY cd.id ASC
    `);
    
    res.json({
      companyDetails,
      companies,
      joinTest,
      message: 'Debug information retrieved',
      analysis: {
        totalCompanies: companies.length,
        totalCompanyDetails: companyDetails.length,
        totalJoinTest: joinTest.length
      }
    });
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Populate sample company details for existing companies
router.post('/populate-sample-details', async (req, res) => {
  try {
    // Get all companies
    const [companies] = await db.query('SELECT * FROM company_details');
    
    let results = [];
    
    for (const company of companies) {
      // Check if details already exist
      const [existing] = await db.query('SELECT id FROM company_details WHERE company_name = ?', [company.name]);
      
      if (existing.length === 0) {
        // Create sample details
        const sampleDetails = {
          company_name: company.company_name,
          address: `Sample address for ${company.company_name}`,
          city: 'Sample City',
          email: `${company.company_name.toLowerCase()}@example.com`,
          pan: `PAN${company.id.toString().padStart(5, '0')}`,
          bank_name: 'Sample Bank',
          account_number: `ACC${company.id.toString().padStart(8, '0')}`,
          ifsc: 'SMPL0000001',
          branch: 'Main Branch',
          account_name: `${company.company_name} Account`
        };
        
        await db.query(`
          INSERT INTO company_details 
          (company_name, address, city, email, pan, bank_name, account_number, ifsc, branch, account_name)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `, [
          sampleDetails.company_name,
          sampleDetails.address,
          sampleDetails.city,
          sampleDetails.email,
          sampleDetails.pan,
          sampleDetails.bank_name,
          sampleDetails.account_number,
          sampleDetails.ifsc,
          sampleDetails.branch,
          sampleDetails.account_name
        ]);
        
        results.push(`Created details for ${company.company_name}`);
      } else {
        results.push(`Details already exist for ${company.company_name}`);
      }
    }
    
    res.json({
      message: 'Sample company details populated',
      results,
      totalCompanies: companies.length
    });
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get company details by company name
router.get('/details/:companyName', async (req, res) => {
  const { companyName } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT * FROM company_details 
      WHERE company_name = ?
    `, [companyName]);
    
    if (rows.length === 0) {
      return res.json({
        company_name: companyName,
        address: null,
        city: null,
        email: null,
        pan: null,
        bank_name: null,
        account_number: null,
        ifsc: null,
        branch: null,
        account_name: null,
        message: 'No additional details found for this company'
      });
    }
    
    res.json(rows[0]);
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add or update company details
router.post('/details', async (req, res) => {
  const {
    company_name,
    address,
    city,
    email,
    pan,
    gst,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name: accountName
  } = req.body;
  
  if (!company_name) {
    return res.status(400).json({ message: 'Company name is required' });
  }
  
  try {
    // Check if company details already exist
    const [existing] = await db.query('SELECT id FROM company_details WHERE company_name = ?', [company_name]);
    
    if (existing.length > 0) {
      // Update existing
      await db.query(`
        UPDATE company_details 
        SET address = ?, city = ?, email = ?, pan = ?, gst = ?, bank_name = ?, 
            account_number = ?, ifsc = ?, branch = ?, account_name = ?
        WHERE company_name = ?
      `, [address, city, email, pan, gst, bank_name, account_number, ifsc, branch, accountName, company_name]);
      
      res.json({ message: 'Company details updated successfully' });
    } else {
      // Insert new
      await db.query(`
        INSERT INTO company_details 
        (company_name, address, city, email, pan, gst, bank_name, account_number, ifsc, branch, account_name)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `, [company_name, address, city, email, pan, gst, bank_name, account_number, ifsc, branch, accountName]);
      
      res.json({ message: 'Company details added successfully' });
    }
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 