const express = require('express');
const router = express.Router();
let db;

function setDb(pool) { 
  db = pool; 
}

// Get all destinations
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT * FROM destinations ORDER BY name ASC');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching destinations:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get destination by ID
router.get('/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const [rows] = await db.query('SELECT * FROM destinations WHERE id = ?', [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Destination not found' });
    }
    
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching destination:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add new destination
router.post('/', async (req, res) => {
  try {
    const { name, location } = req.body;
    
    if (!name || !location) {
      return res.status(400).json({ message: 'Name and location are required' });
    }
    
    const [result] = await db.query(
      'INSERT INTO destinations (name, location) VALUES (?, ?)',
      [name, location]
    );
    
    res.status(201).json({ 
      message: 'Destination added successfully',
      id: result.insertId 
    });
  } catch (err) {
    console.error('Error adding destination:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update destination
router.put('/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const { name, location } = req.body;
    
    if (!name || !location) {
      return res.status(400).json({ message: 'Name and location are required' });
    }
    
    const [result] = await db.query(
      'UPDATE destinations SET name = ?, location = ? WHERE id = ?',
      [name, location, id]
    );
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Destination not found' });
    }
    
    res.json({ message: 'Destination updated successfully' });
  } catch (err) {
    console.error('Error updating destination:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete destination
router.delete('/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    const [result] = await db.query('DELETE FROM destinations WHERE id = ?', [id]);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Destination not found' });
    }
    
    res.json({ message: 'Destination deleted successfully' });
  } catch (err) {
    console.error('Error deleting destination:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Auto-add destination from invoice creation
router.post('/auto-add', async (req, res) => {
  try {
    const { name, location } = req.body;
    
    if (!name || !location) {
      return res.status(400).json({ message: 'Name and location are required' });
    }
    
    // Check if destination already exists
    const [existing] = await db.query(
      'SELECT id FROM destinations WHERE name = ? AND location = ?',
      [name, location]
    );
    
    if (existing.length > 0) {
      return res.json({ 
        message: 'Destination already exists',
        id: existing[0].id,
        exists: true
      });
    }
    
    // Add new destination
    const [result] = await db.query(
      'INSERT INTO destinations (name, location) VALUES (?, ?)',
      [name, location]
    );
    
    res.status(201).json({ 
      message: 'Destination auto-added successfully',
      id: result.insertId,
      exists: false
    });
  } catch (err) {
    console.error('Error auto-adding destination:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb };
