const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// Get all diesel purchases
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT dp.*, v.number as vehicle_number, v.name as vehicle_name, b.bunk_name
      FROM diesel_purchases dp 
      LEFT JOIN vehicles v ON dp.vehicle_id = v.id
      LEFT JOIN bunks b ON dp.bunk_id = b.id
      ORDER BY dp.created_at ASC
    `);
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching diesel purchases:', err);
    res.status(500).json({ error: 'Failed to fetch diesel purchases' });
  }
});

// Get all diesel purchases for a given driver_name
router.get('/by-driver/:driverName', async (req, res) => {
  const { driverName } = req.params;
  try {
    console.log('Fetching diesel purchases for driver:', driverName);
    
    // First check if driver_name column exists
    const [columns] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'diesel_purchases'
      AND COLUMN_NAME = 'driver_name'
    `);

    if (columns.length === 0) {
      console.log('driver_name column does not exist, adding it...');
      try {
        await db.query(`
          ALTER TABLE diesel_purchases 
          ADD COLUMN driver_name VARCHAR(255) NULL 
          AFTER vehicle_id
        `);
        console.log('✅ Added driver_name column to diesel_purchases table');
      } catch (alterError) {
        console.log('Error adding column:', alterError.message);
        // Continue with empty results if column can't be added
        return res.json([]);
      }
    }
    
    // Now try to fetch diesel purchases
    const [rows] = await db.query(`
      SELECT dp.*, v.number as vehicle_number, v.name as vehicle_name, b.bunk_name
      FROM diesel_purchases dp 
      LEFT JOIN vehicles v ON dp.vehicle_id = v.id
      LEFT JOIN bunks b ON dp.bunk_id = b.id
      WHERE LOWER(TRIM(dp.driver_name)) = LOWER(TRIM(?))
      ORDER BY dp.bill_date ASC
    `, [driverName]);
    
    console.log('Found diesel purchases for driver:', rows.length);
    console.log('Query result:', rows);
    
    // If no results, let's check what driver names exist
    if (rows.length === 0) {
      const [allDrivers] = await db.query(`
        SELECT DISTINCT driver_name, COUNT(*) as count
        FROM diesel_purchases 
        WHERE driver_name IS NOT NULL
        GROUP BY driver_name
        ORDER BY count DESC
      `);
      console.log('All driver names in diesel_purchases:', allDrivers);
      console.log('Searching for driver name:', driverName);
    }
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching diesel purchases by driver:', err);
    res.status(500).json({ error: 'Failed to fetch diesel purchases by driver: ' + err.message });
  }
});

// Get diesel purchase by ID
router.get('/:id', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT dp.*, v.number as vehicle_number, v.name as vehicle_name, b.bunk_name
      FROM diesel_purchases dp 
      LEFT JOIN vehicles v ON dp.vehicle_id = v.id
      LEFT JOIN bunks b ON dp.bunk_id = b.id
      WHERE dp.id = ?
    `, [req.params.id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ error: 'Diesel purchase not found' });
    }
    
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching diesel purchase:', err);
    res.status(500).json({ error: 'Failed to fetch diesel purchase' });
  }
});

// Create new diesel purchase
router.post('/', async (req, res) => {
  try {
    const { bill_no, bill_date, vehicle_id, driver_name, litres, rate, bunk_id, total_amount } = req.body;
    
    // Validate required fields
    if (!bill_no || !bill_date || !vehicle_id || !litres || !rate) {
      return res.status(400).json({ error: 'All fields are required' });
    }
    
    // Use provided total_amount or calculate it
    const final_total_amount = total_amount ? parseFloat(total_amount) : (parseFloat(litres) * parseFloat(rate));
    
    // Insert diesel purchase record
    const [result] = await db.query(
      'INSERT INTO diesel_purchases (bill_no, bill_date, vehicle_id, driver_name, litres, rate, total_amount, bunk_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)',
      [bill_no, bill_date, vehicle_id, driver_name || null, litres, rate, final_total_amount, bunk_id || null]
    );
    
    res.status(201).json({ 
      message: 'Diesel purchase created successfully', 
      id: result.insertId 
    });
  } catch (err) {
    console.error('Error creating diesel purchase:', err);
    res.status(500).json({ error: 'Failed to create diesel purchase' });
  }
});

// Update diesel purchase
router.put('/:id', async (req, res) => {
  try {
    const { bill_no, bill_date, vehicle_id, driver_name, litres, rate, bunk_id, total_amount } = req.body;
    
    // Validate required fields
    if (!bill_no || !bill_date || !vehicle_id || !litres || !rate) {
      return res.status(400).json({ error: 'All fields are required' });
    }
    
    // Use provided total_amount or calculate it
    const final_total_amount = total_amount ? parseFloat(total_amount) : (parseFloat(litres) * parseFloat(rate));
    
    // Update diesel purchase record
    await db.query(
      'UPDATE diesel_purchases SET bill_no = ?, bill_date = ?, vehicle_id = ?, driver_name = ?, litres = ?, rate = ?, total_amount = ?, bunk_id = ? WHERE id = ?',
      [bill_no, bill_date, vehicle_id, driver_name || null, litres, rate, final_total_amount, bunk_id || null, req.params.id]
    );
    
    res.json({ message: 'Diesel purchase updated successfully' });
  } catch (err) {
    console.error('Error updating diesel purchase:', err);
    res.status(500).json({ error: 'Failed to update diesel purchase' });
  }
});

// Delete diesel purchase
router.delete('/:id', async (req, res) => {
  try {
    await db.query('DELETE FROM diesel_purchases WHERE id = ?', [req.params.id]);
    res.json({ message: 'Diesel purchase deleted successfully' });
  } catch (err) {
    console.error('Error deleting diesel purchase:', err);
    res.status(500).json({ error: 'Failed to delete diesel purchase' });
  }
});

module.exports = { router, setDb }; 