const express = require('express');
const router = express.Router();
const mysql = require('mysql2/promise');

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// Get all drivers
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT * FROM drivers ORDER BY created_at ASC');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching drivers:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get single driver by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  
  try {
    const [rows] = await db.query('SELECT * FROM drivers WHERE id = ?', [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Driver not found' });
    }
    
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching driver:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add new driver
router.post('/', async (req, res) => {
  const { name, phone, altPhone } = req.body;
  
  if (!name || !phone) {
    return res.status(400).json({ message: 'Driver name and phone are required' });
  }

  // Validate phone number format
  const phoneRegex = /^\+?\d{10,15}$/;
  if (!phoneRegex.test(phone)) {
    return res.status(400).json({ message: 'Invalid phone number format' });
  }

  if (altPhone && !phoneRegex.test(altPhone)) {
    return res.status(400).json({ message: 'Invalid alternate phone number format' });
  }

  try {
    const [result] = await db.query(
      'INSERT INTO drivers (name, phone, alt_phone) VALUES (?, ?, ?)',
      [name, phone, altPhone || null]
    );
    
    res.status(201).json({ 
      message: 'Driver added successfully',
      id: result.insertId,
      driver: { id: result.insertId, name, phone, altPhone }
    });
  } catch (err) {
    console.error('Error adding driver:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update driver
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const { name, phone, altPhone } = req.body;
  
  if (!name || !phone) {
    return res.status(400).json({ message: 'Driver name and phone are required' });
  }

  // Validate phone number format
  const phoneRegex = /^\+?\d{10,15}$/;
  if (!phoneRegex.test(phone)) {
    return res.status(400).json({ message: 'Invalid phone number format' });
  }

  if (altPhone && !phoneRegex.test(altPhone)) {
    return res.status(400).json({ message: 'Invalid alternate phone number format' });
  }

  try {
    // Check if driver exists
    const [existing] = await db.query('SELECT id FROM drivers WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Driver not found' });
    }

    await db.query(
      'UPDATE drivers SET name = ?, phone = ?, alt_phone = ? WHERE id = ?',
      [name, phone, altPhone || null, id]
    );
    
    res.json({ message: 'Driver updated successfully' });
  } catch (err) {
    console.error('Error updating driver:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete driver
router.delete('/:id', async (req, res) => {
  const { id } = req.params;

  try {
    const [result] = await db.query('DELETE FROM drivers WHERE id = ?', [id]);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Driver not found' });
    }
    
    res.json({ message: 'Driver deleted successfully' });
  } catch (err) {
    console.error('Error deleting driver:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 