const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// Get all expenses with related data
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT 
        e.id,
        e.vehicle_id,
        e.driver_id,
        e.type,
        e.litre,
        e.rate,
        e.amount,
        e.expense_date,
        e.created_at,
        v.number as vehicle_number,
        v.name as vehicle_name,
        d.name as driver_name,
        d.phone as driver_phone
      FROM expenses e
      LEFT JOIN vehicles v ON e.vehicle_id = v.id
      LEFT JOIN drivers d ON e.driver_id = d.id
      ORDER BY e.created_at ASC
    `);
    
    // Convert timestamps to Indian time
    const expenses = rows.map(expense => {
      const createdAt = new Date(expense.created_at);
      const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
      
      // Format expense_date to show exactly as stored in database
      let formattedExpenseDate = '';
      if (expense.expense_date) {
        if (typeof expense.expense_date === 'string' && expense.expense_date.includes('-')) {
          formattedExpenseDate = expense.expense_date.slice(0, 10);
        } else {
          const date = new Date(expense.expense_date);
          const year = date.getFullYear();
          const month = String(date.getMonth() + 1).padStart(2, '0');
          const day = String(date.getDate()).padStart(2, '0');
          formattedExpenseDate = `${year}-${month}-${day}`;
        }
      }
      
      // Format created_at to show only date (DD/MM/YYYY)
      let formattedCreatedAt = '';
      if (expense.created_at) {
        const createdDate = new Date(expense.created_at);
        const day = createdDate.getDate().toString().padStart(2, '0');
        const month = (createdDate.getMonth() + 1).toString().padStart(2, '0');
        const year = createdDate.getFullYear();
        formattedCreatedAt = `${day}/${month}/${year}`;
      }
      
      // Format type for display
      const typeDisplay = {
        'fuel': 'Fuel',
        'driver_charge': 'Driver Charge',
        'toll': 'Toll'
      }[expense.type] || expense.type;
      
      // Format amount display for fuel
      let amountDisplay = `₹${parseFloat(expense.amount).toFixed(2)}`;
      if (expense.type === 'fuel' && expense.litre && expense.rate) {
        amountDisplay = `${expense.litre}L × ₹${expense.rate} = ₹${parseFloat(expense.amount).toFixed(2)}`;
      }
      
      return {
        ...expense,
        type: expense.type,
        type_display: typeDisplay,
        amount_display: amountDisplay,
        expense_date: formattedExpenseDate,
        created_at: formattedCreatedAt,
        driver_name: expense.driver_name || 'Not Assigned',
        vehicle_number: expense.vehicle_number || 'N/A',
        vehicle_name: expense.vehicle_name || 'N/A'
      };
    });
    
    res.json(expenses);
  } catch (err) {
    console.error('Error fetching expenses:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get single expense by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    const [rows] = await db.query(`
      SELECT 
        e.id,
        e.vehicle_id,
        e.driver_id,
        e.type,
        e.litre,
        e.rate,
        e.amount,
        e.expense_date,
        e.created_at,
        v.number as vehicle_number,
        v.name as vehicle_name,
        d.name as driver_name,
        d.phone as driver_phone
      FROM expenses e
      LEFT JOIN vehicles v ON e.vehicle_id = v.id
      LEFT JOIN drivers d ON e.driver_id = d.id
      WHERE e.id = ?
    `, [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Expense not found' });
    }
    
    const expense = rows[0];
    
    // Convert timestamps to Indian time
    const createdAt = new Date(expense.created_at);
    const indianCreatedAt = new Date(createdAt.toLocaleString("en-US", {timeZone: "Asia/Kolkata"}));
    
    // Format expense_date
    let formattedExpenseDate = '';
    if (expense.expense_date) {
      if (typeof expense.expense_date === 'string' && expense.expense_date.includes('-')) {
        formattedExpenseDate = expense.expense_date.slice(0, 10);
      } else {
        const date = new Date(expense.expense_date);
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        formattedExpenseDate = `${year}-${month}-${day}`;
      }
    }
    
    expense.expense_date = formattedExpenseDate;
    expense.created_at = indianCreatedAt.toISOString();
    
    res.json(expense);
  } catch (err) {
    console.error('Error fetching expense:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Create new expense
router.post('/', async (req, res) => {
  const {
    vehicle_id,
    driver_id,
    type,
    litre,
    rate,
    amount,
    expense_date
  } = req.body;
  
  // Validation
  if (!vehicle_id || !type || !amount || !expense_date) {
    return res.status(400).json({ 
      message: 'Vehicle, type, amount, and expense date are required' 
    });
  }

  // Additional validation for fuel type
  if (type === 'fuel' && (!litre || !rate)) {
    return res.status(400).json({ 
      message: 'Litre and rate are required for fuel expenses' 
    });
  }

  try {
    // Check if vehicle exists
    const [vehicleExists] = await db.query('SELECT id FROM vehicles WHERE id = ?', [vehicle_id]);
    if (vehicleExists.length === 0) {
      return res.status(400).json({ message: 'Vehicle not found' });
    }

    // Check if driver exists (if provided)
    if (driver_id) {
      const [driverExists] = await db.query('SELECT id FROM drivers WHERE id = ?', [driver_id]);
      if (driverExists.length === 0) {
        return res.status(400).json({ message: 'Driver not found' });
      }
    }

    const [result] = await db.query(`
      INSERT INTO expenses (vehicle_id, driver_id, type, litre, rate, amount, expense_date)
      VALUES (?, ?, ?, ?, ?, ?, ?)
    `, [vehicle_id, driver_id, type, litre ? parseFloat(litre) : null, rate ? parseFloat(rate) : null, parseFloat(amount), expense_date]);
    
    res.status(201).json({ 
      message: 'Expense created successfully',
      id: result.insertId
    });
  } catch (err) {
    console.error('Error creating expense:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update expense
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const {
    vehicle_id,
    driver_id,
    type,
    litre,
    rate,
    amount,
    expense_date
  } = req.body;
  
  // Validation
  if (!vehicle_id || !type || !amount || !expense_date) {
    return res.status(400).json({ 
      message: 'Vehicle, type, amount, and expense date are required' 
    });
  }

  // Additional validation for fuel type
  if (type === 'fuel' && (!litre || !rate)) {
    return res.status(400).json({ 
      message: 'Litre and rate are required for fuel expenses' 
    });
  }

  try {
    // Check if expense exists
    const [existing] = await db.query('SELECT id FROM expenses WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Expense not found' });
    }

    // Check if vehicle exists
    const [vehicleExists] = await db.query('SELECT id FROM vehicles WHERE id = ?', [vehicle_id]);
    if (vehicleExists.length === 0) {
      return res.status(400).json({ message: 'Vehicle not found' });
    }

    // Check if driver exists (if provided)
    if (driver_id) {
      const [driverExists] = await db.query('SELECT id FROM drivers WHERE id = ?', [driver_id]);
      if (driverExists.length === 0) {
        return res.status(400).json({ message: 'Driver not found' });
      }
    }

    await db.query(`
      UPDATE expenses SET 
        vehicle_id = ?, driver_id = ?, type = ?, litre = ?, rate = ?, amount = ?, expense_date = ?
      WHERE id = ?
    `, [vehicle_id, driver_id, type, litre ? parseFloat(litre) : null, rate ? parseFloat(rate) : null, parseFloat(amount), expense_date, id]);
    
    res.json({ message: 'Expense updated successfully' });
  } catch (err) {
    console.error('Error updating expense:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete expense
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    // Check if expense exists
    const [existing] = await db.query('SELECT id FROM expenses WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Expense not found' });
    }

    await db.query('DELETE FROM expenses WHERE id = ?', [id]);
    res.json({ message: 'Expense deleted successfully' });
  } catch (err) {
    console.error('Error deleting expense:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 