const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// Simple endpoint to fix receipts table - make company_id nullable
router.get('/fix-receipts-now', async (req, res) => {
  try {
    console.log('🔧 Fixing receipts table...');
    
    // Just make it nullable - simple and direct
    await db.query('ALTER TABLE receipts MODIFY COLUMN company_id INT NULL');
    
    res.json({
      success: true,
      message: '✅ Fixed! Company is now optional in receipts table.'
    });
  } catch (error) {
    if (error.code === 'ER_BAD_FIELD_ERROR') {
      res.json({
        success: true,
        message: '✅ Column already allows NULL - nothing to fix!'
      });
    } else {
      res.status(500).json({
        success: false,
        error: error.message
      });
    }
  }
});

// Endpoint to fix receipts table company_id column
router.get('/receipts-company', async (req, res) => {
  try {
    const results = {
      steps: [],
      success: false
    };

    // Check if company_details table exists
    const [tables] = await db.query(`
      SELECT TABLE_NAME 
      FROM INFORMATION_SCHEMA.TABLES 
      WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'company_details'
    `);
    
    if (tables.length === 0) {
      results.steps.push('ERROR: company_details table does not exist');
      return res.json(results);
    }
    results.steps.push('✅ company_details table exists');

    // Check current column configuration
    const [columns] = await db.query(`
      SELECT IS_NULLABLE, COLUMN_TYPE
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'receipts'
      AND COLUMN_NAME = 'company_id'
    `);
    
    if (columns.length === 0) {
      results.steps.push('ERROR: company_id column does not exist in receipts table');
      return res.json(results);
    }
    
    results.steps.push(`Current: ${columns[0].COLUMN_TYPE}, NULL: ${columns[0].IS_NULLABLE}`);

    // Check for existing foreign keys on company_id
    const [fks] = await db.query(`
      SELECT CONSTRAINT_NAME, REFERENCED_TABLE_NAME
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE
      WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = 'receipts'
      AND COLUMN_NAME = 'company_id'
      AND REFERENCED_TABLE_NAME IS NOT NULL
    `);
    
    // Drop existing foreign keys
    for (const fk of fks) {
      results.steps.push(`Dropping FK: ${fk.CONSTRAINT_NAME} -> ${fk.REFERENCED_TABLE_NAME}`);
      try {
        await db.query(`ALTER TABLE receipts DROP FOREIGN KEY ${fk.CONSTRAINT_NAME}`);
        results.steps.push('✅ FK dropped');
      } catch (error) {
        results.steps.push(`WARNING: ${error.message}`);
      }
    }

    // Make column nullable
    if (columns[0].IS_NULLABLE === 'NO') {
      results.steps.push('Making company_id nullable...');
      await db.query(`ALTER TABLE receipts MODIFY COLUMN company_id INT NULL`);
      results.steps.push('✅ Column is now nullable');
    } else {
      results.steps.push('✅ Column already nullable');
    }

    // Add foreign key to company_details
    results.steps.push('Adding FK to company_details...');
    try {
      await db.query(`
        ALTER TABLE receipts 
        ADD CONSTRAINT fk_receipts_company_details 
        FOREIGN KEY (company_id) REFERENCES company_details(id) 
        ON DELETE SET NULL 
        ON UPDATE CASCADE
      `);
      results.steps.push('✅ FK added to company_details');
    } catch (error) {
      if (error.code === 'ER_DUP_KEYNAME') {
        results.steps.push('ℹ️ FK already exists');
      } else {
        results.steps.push(`WARNING: ${error.message}`);
      }
    }

    results.success = true;
    results.steps.push('✅ ✅ ✅ DONE! Company is now optional and references company_details');
    res.json(results);

  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message,
      stack: error.stack
    });
  }
});

module.exports = { router, setDb };

