



const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// Get all payment entries
router.get('/', async (req, res) => {
  try {
    console.log('Fetching payment entries...');
    console.log('Database connection:', db ? 'Connected' : 'Not connected');
    
    // First test a simple query
    const [testRows] = await db.query('SELECT COUNT(*) as count FROM payment_entries');
    console.log('Payment entries count:', testRows[0].count);
    
    // Then run the main query with bunk information
    const [rows] = await db.query(`
      SELECT 
        pe.*, 
        pe.driver_name,
        b.bunk_name,
        c.name as company_name,
        cust.name as customer_name,
        CASE 
          WHEN pe.account_type_bank_name IS NOT NULL THEN pe.account_type_bank_name
          ELSE 'N/A'
        END as display_bank_name
      FROM payment_entries pe
      LEFT JOIN bunks b ON pe.bunk_id = b.id
      LEFT JOIN companies c ON pe.company_id = c.id
      LEFT JOIN customers cust ON pe.customer_id = cust.id
      ORDER BY pe.created_at ASC
    `);
    
    console.log('Payment entries query successful, found', rows.length, 'entries');
    
    // Debug: Log sample payment data to see bunk_id values
    if (rows.length > 0) {
      console.log('Sample payment entry with bunk data:', {
        id: rows[0].id,
        bunk_id: rows[0].bunk_id,
        bunk_name: rows[0].bunk_name,
        bill_no: rows[0].bill_no,
        account_type_bank_name: rows[0].account_type_bank_name
      });
      
      // Count payments with and without bunk_id
      const paymentsWithBunkId = rows.filter(p => p.bunk_id);
      const paymentsWithoutBunkId = rows.filter(p => !p.bunk_id);
      console.log(`Payments with bunk_id: ${paymentsWithBunkId.length}`);
      console.log(`Payments without bunk_id: ${paymentsWithoutBunkId.length}`);
      
      if (paymentsWithoutBunkId.length > 0) {
        console.log('Sample payment without bunk_id:', {
          id: paymentsWithoutBunkId[0].id,
          bill_no: paymentsWithoutBunkId[0].bill_no,
          account_type_bank_name: paymentsWithoutBunkId[0].account_type_bank_name
        });
      }
    }
    
    res.json(rows);
  } catch (err) {
    console.error('Error fetching payment entries:', err);
    console.error('Error details:', err.message);
    console.error('Error code:', err.code);
    console.error('SQL State:', err.sqlState);
    res.status(500).json({ error: 'Failed to fetch payment entries', details: err.message });
  }
});

// Get banks for dropdown
router.get('/banks/list', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT id, bank_name FROM banks ORDER BY bank_name');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching banks:', err);
    res.status(500).json({ error: 'Failed to fetch banks' });
  }
});

// Get drivers for dropdown
router.get('/drivers/list', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT id, name FROM drivers ORDER BY name');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching drivers:', err);
    res.status(500).json({ error: 'Failed to fetch drivers' });
  }
});

// Get vehicles for dropdown
router.get('/vehicles/list', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT id, number, name FROM vehicles ORDER BY number');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching vehicles:', err);
    res.status(500).json({ error: 'Failed to fetch vehicles' });
  }
});

// Add endpoint to get latest bill_no
router.get('/latest-bill-no', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT bill_no FROM payment_entries ORDER BY id DESC LIMIT 1');
    if (rows.length === 0) {
      return res.json({ bill_no: null });
    }
    res.json({ bill_no: rows[0].bill_no });
  } catch (err) {
    console.error('Error fetching latest bill_no:', err);
    res.status(500).json({ error: 'Failed to fetch latest bill_no' });
  }
});

// Get payment entry by ID
router.get('/:id', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT 
        pe.*, 
        pe.driver_name,
        CASE 
          WHEN pe.account_type_bank_name IS NOT NULL THEN pe.account_type_bank_name
          ELSE 'N/A'
        END as display_bank_name
      FROM payment_entries pe
      WHERE pe.id = ?
    `, [req.params.id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ error: 'Payment entry not found' });
    }
    
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching payment entry:', err);
    res.status(500).json({ error: 'Failed to fetch payment entry' });
  }
});

// Create new payment entry
router.post('/', async (req, res) => {
  try {
    const { 
      bill_no, 
      bill_date, 
      account_type_bank_name, 
      bank_id,
      driver_id,
      driver_name, 
      vehicle_no, 
      bunk_id,
    company_id = null,
    customer_id = null,
    selected_expense_id,
    expense_amount,
      total,
    note
    } = req.body;
    
    // Debug: Log all received data
    console.log('📥 Received payment data:', {
      bill_no,
      bill_date,
      account_type_bank_name,
      bank_id,
      driver_id,
      driver_name,
      vehicle_no,
      bunk_id,
      company_id,
      customer_id,
      selected_expense_id,
      expense_amount,
      total,
      note,
      customer_id_type: typeof customer_id,
      customer_id_value: customer_id
    });
    
    // CRITICAL: Detailed customer_id analysis
    console.log('🚨 DETAILED CUSTOMER_ID ANALYSIS:', {
      raw_customer_id: customer_id,
      type: typeof customer_id,
      is_null: customer_id === null,
      is_undefined: customer_id === undefined,
      is_empty_string: customer_id === '',
      is_string_null: customer_id === 'null',
      is_falsy: !customer_id,
      string_length: customer_id ? customer_id.toString().length : 'N/A',
      parsed_int: customer_id ? parseInt(customer_id) : 'N/A'
    });
    
    console.log('🔍 All IDs received:', {
      bank_id: bank_id,
      driver_id: driver_id,
      bunk_id: bunk_id,
      company_id: company_id,
      customer_id: customer_id,
      selected_expense_id: selected_expense_id
    });
    
    // Validate required fields
    if (!bill_no || !bill_date || !account_type_bank_name) {
      console.log('❌ Validation failed - missing required fields:', {
        bill_no: !!bill_no,
        bill_date: !!bill_date,
        account_type_bank_name: !!account_type_bank_name
      });
      return res.status(400).json({ 
        error: 'Bill number, bill date, and account type are required',
        received: { bill_no, bill_date, account_type_bank_name }
      });
    }
    
    // Driver name is optional - set default value if null to avoid database constraint
    const finalDriverName = driver_name || 'N/A';
    const finalVehicleNo = vehicle_no || 'N/A';
    
    // Validate driver_id if provided
    let safeDriverId = null;
    if (driver_id && driver_id !== null && driver_id !== '') {
      try {
        console.log('🔍 Validating driver_id:', driver_id);
        const [driverCheck] = await db.query('SELECT id, name FROM drivers WHERE id = ?', [driver_id]);
        if (driverCheck.length > 0) {
          safeDriverId = driver_id;
          console.log('✅ Driver validation passed:', driverCheck[0]);
        } else {
          console.log('⚠️ Driver not found, setting to NULL:', driver_id);
        }
      } catch (error) {
        console.log('⚠️ Error validating driver, setting to NULL:', error.message);
      }
    }
    
    // Validate bunk_id if provided
    let safeBunkId = null;
    if (bunk_id && bunk_id !== null && bunk_id !== '') {
      try {
        console.log('🔍 Validating bunk_id:', bunk_id);
        const [bunkCheck] = await db.query('SELECT id, bunk_name FROM bunks WHERE id = ?', [bunk_id]);
        if (bunkCheck.length > 0) {
          safeBunkId = bunk_id;
          console.log('✅ Bunk validation passed:', bunkCheck[0]);
        } else {
          console.log('⚠️ Bunk not found, setting to NULL:', bunk_id);
        }
      } catch (error) {
        console.log('⚠️ Error validating bunk, setting to NULL:', error.message);
      }
    }
    
    // Validate bank_id if provided
    let safeBankId = null;
    if (bank_id && bank_id !== null && bank_id !== '') {
      try {
        console.log('🔍 Validating bank_id:', bank_id);
        const [bankCheck] = await db.query('SELECT id, bank_name FROM banks WHERE id = ?', [bank_id]);
        if (bankCheck.length > 0) {
          safeBankId = bank_id;
          console.log('✅ Bank validation passed:', bankCheck[0]);
        } else {
          console.log('⚠️ Bank not found, setting to NULL:', bank_id);
        }
      } catch (error) {
        console.log('⚠️ Error validating bank, setting to NULL:', error.message);
      }
    }
    
    // FORCE customer_id to be saved - BYPASS ALL VALIDATION
    let safeCustomerId = null;
    console.log('🚨 BYPASSING CUSTOMER VALIDATION - FORCING SAVE');
    console.log('🚨 Raw customer_id received:', customer_id, 'Type:', typeof customer_id);
    
    // Accept ANY non-null, non-undefined, non-empty value
    if (customer_id !== null && customer_id !== undefined && customer_id !== '' && customer_id !== 'null') {
      safeCustomerId = customer_id;
      console.log('✅ FORCING customer_id to be saved:', safeCustomerId, 'Type:', typeof safeCustomerId);
    } else {
      console.log('⚠️ customer_id is null/undefined/empty/string-null, setting to null');
      safeCustomerId = null;
    }
    
    console.log('🎯 Final customer_id to be saved (FORCED):', safeCustomerId);
    
    // Validate company_id if provided
    let safeCompanyId = null;
    if (company_id && company_id !== null && company_id !== '') {
      try {
        console.log('🔍 Validating company_id:', company_id);
        const [companyCheck] = await db.query('SELECT id, company_name FROM company_details WHERE id = ?', [company_id]);
        console.log('📋 Company check result:', companyCheck);
        
        if (companyCheck.length === 0) {
          console.log('❌ Company not found:', company_id);
          
          // Get available companies for debugging
          const [availableCompanies] = await db.query('SELECT id, company_name FROM company_details LIMIT 5');
          console.log('📋 Available companies:', availableCompanies);
          
          return res.status(400).json({ 
            error: 'Selected company not found', 
            company_id: company_id,
            available_companies: availableCompanies,
            message: 'Please select a valid company from the list'
          });
        }
        safeCompanyId = company_id;
        console.log('✅ Company validation passed:', companyCheck[0]);
      } catch (error) {
        console.error('❌ Error validating company:', error);
        return res.status(400).json({ 
          error: 'Invalid company selection', 
          company_id: company_id 
        });
      }
    }
    
    // Validate selected expense type if provided
    if (selected_expense_id) {
      console.log('🔍 Validating expense type ID:', selected_expense_id);
      
      // Check if it exists in expense_types table only
      const [expenseTypeCheck] = await db.query('SELECT id, name FROM expense_types WHERE id = ?', [selected_expense_id]);
      
      console.log('📋 Expense type check result:', expenseTypeCheck);
      
      if (expenseTypeCheck.length === 0) {
        console.log('❌ Expense type not found:', selected_expense_id);
        return res.status(400).json({ error: 'Selected expense type not found', expense_id: selected_expense_id });
      }
      console.log('✅ Expense type validation passed:', expenseTypeCheck[0]);
    }

    // Note: selected_expense_type_id validation removed as this column doesn't exist in database
    
    // Build dynamic query columns and values
    const baseColumns = [
      'bill_no', 'bill_date', 'account_type_bank_name', 'bank_id', 'driver_id',
      'driver_name', 'vehicle_no', 'bunk_id', 'company_id', 'customer_id'
    ];
    
    const baseParams = [
      bill_no, bill_date, account_type_bank_name, safeBankId, safeDriverId,
      finalDriverName, finalVehicleNo, safeBunkId, safeCompanyId, safeCustomerId
    ];
    
    // Add new expense columns
    const expenseColumns = [
      'selected_expense_id', 'expense_amount', 'total', 'note'
    ];
    
    // Simplified approach: just use the selected_expense_id as is
    let finalSelectedExpenseId = null;
    console.log('🔍 Processing selected_expense_id:', selected_expense_id, 'Type:', typeof selected_expense_id);
    
    if (selected_expense_id && selected_expense_id !== 'null' && selected_expense_id !== null && selected_expense_id !== '') {
      finalSelectedExpenseId = selected_expense_id;
      console.log('✅ Using selected_expense_id directly:', finalSelectedExpenseId);
          } else {
      console.log('⚠️ selected_expense_id is null or empty');
    }
    
    console.log('🎯 Final selected expense ID:', finalSelectedExpenseId);
    
    const expenseParams = [
      finalSelectedExpenseId,
      parseFloat(expense_amount || 0), 
      parseFloat(total || 0),
      note && note.trim() !== '' ? note : 'N/A'
    ];
    
    // Combine all columns and parameters
    const allColumns = [...baseColumns, ...expenseColumns];
    const allParams = [...baseParams, ...expenseParams];
    
    // Build the INSERT query
    const placeholders = allColumns.map(() => '?').join(', ');
    const insertQuery = `INSERT INTO payment_entries (${allColumns.join(', ')}) VALUES (${placeholders})`;
    
    console.log('🔧 INSERT query:', insertQuery);
    console.log('📊 All columns:', allColumns);
    console.log('💰 All params:', allParams);
    console.log('🎯 Final IDs being inserted:', {
      'Column 0 - bill_no': allParams[0],
      'Column 1 - bill_date': allParams[1],
      'Column 2 - account_type_bank_name': allParams[2],
      'Column 3 - bank_id': allParams[3],
      'Column 4 - driver_id': allParams[4],
      'Column 5 - driver_name': allParams[5],
      'Column 6 - vehicle_no': allParams[6],
      'Column 7 - bunk_id': allParams[7],
      'Column 8 - company_id': allParams[8],
      'Column 9 - customer_id': allParams[9],
      'Column 10 - selected_expense_id': allParams[10],
      'Column 11 - expense_amount': allParams[11],
      'Column 12 - total': allParams[12],
      'Column 13 - note': allParams[13]
    });
    
    console.log('🔍 DETAILED customer_id check:', {
      received_customer_id: customer_id,
      safeCustomerId: safeCustomerId,
      customer_id_in_params: allParams[9],
      customer_id_type: typeof allParams[9],
      allParamsLength: allParams.length,
      allParams: allParams
    });
    
    // FORCE customer_id to be saved even if validation fails
    if (customer_id && customer_id !== null && customer_id !== '' && customer_id !== 'null') {
      console.log('🚨 FORCING customer_id to be saved:', customer_id);
      allParams[9] = customer_id; // Force it at position 9
    } else {
      console.log('🚨 customer_id is null/empty, keeping as null');
      allParams[9] = null; // Explicitly set to null
    }
    
    console.log('🚨 FINAL customer_id override:', {
      original_customer_id: customer_id,
      safeCustomerId: safeCustomerId,
      allParams_9_before: allParams[9],
      allParams_9_after: allParams[9]
    });
    
    // Debug: Log the payment data being saved
    console.log('🔍 Creating payment entry with data:', {
      bill_no,
      bill_date,
      account_type_bank_name,
      bank_id,
      bank_id_type: typeof bank_id,
      driver_name,
      vehicle_no,
      bunk_id,
      bunk_id_type: typeof bunk_id,
      original_company_id: company_id,
      validated_company_id: safeCompanyId,
      company_id_type: typeof company_id,
      total
    });
    
    // Debug: Log the raw request body
    console.log('📥 Raw request body:', req.body);
    console.log('🏦 Bank ID debugging:', {
      bank_id_from_body: req.body.bank_id,
      bank_id_type: typeof req.body.bank_id,
      bank_id_destructured: bank_id,
      bank_id_destructured_type: typeof bank_id,
      bank_id_final: bank_id || null
    });
    
    // Debug: Log the exact parameters being sent to database
    console.log('🗄️ Database parameters:', {
      insertQuery: insertQuery,
      params: allParams,
      paramsLength: allParams.length,
      bank_id_param: allParams[3], // bank_id should be at index 3
      bunk_id_param: allParams[6]  // bunk_id should be at index 6
    });
    
    // Insert payment entry record
    const [result] = await db.query(insertQuery, allParams);
    
    console.log('✅ Payment entry created successfully with ID:', result.insertId);
    
    // Verify what was saved by reading it back
    const [savedEntry] = await db.query('SELECT * FROM payment_entries WHERE id = ?', [result.insertId]);
    console.log('📋 Saved payment entry verification:', {
      id: result.insertId,
      bank_id: savedEntry[0].bank_id,
      driver_id: savedEntry[0].driver_id,
      driver_name: savedEntry[0].driver_name,
      bunk_id: savedEntry[0].bunk_id,
      company_id: savedEntry[0].company_id,
      customer_id: savedEntry[0].customer_id,
      selected_expense_id: savedEntry[0].selected_expense_id
    });
    
    // CRITICAL: Verify customer_id was saved correctly
    console.log('🚨 CUSTOMER_ID VERIFICATION:', {
      saved_customer_id: savedEntry[0].customer_id,
      saved_customer_id_type: typeof savedEntry[0].customer_id,
      is_null: savedEntry[0].customer_id === null,
      original_customer_id: customer_id,
      match: savedEntry[0].customer_id == customer_id
    });
    
    res.status(201).json({ 
      message: 'Payment entry created successfully', 
      id: result.insertId 
    });
  } catch (err) {
    console.error('Error creating payment entry:', err);
    console.error('Error details:', err.message);
    console.error('Error code:', err.code);
    res.status(500).json({ 
      error: 'Failed to create payment entry', 
      details: err.message,
      code: err.code 
    });
  }
});

// Update payment entry
router.put('/:id', async (req, res) => {
  try {
    const { 
      bill_no, 
      bill_date, 
      account_type_bank_name, 
      bank_id,
      driver_id,
      driver_name, 
      vehicle_no, 
      bunk_id,
      company_id,
      customer_id,
      selected_expense_id,
      expense_amount,
      total,
      note,
      // Legacy fields (keep for backward compatibility)
      driver_advance, 
      unloading_charges, 
      diesel, 
      diesel_litre,
      diesel_rate,
      toll_gate, 
      tyres_purchase, 
      wheel_alignment, 
      servicing_charges, 
      other_expenses, 
      road_tax, 
      insurance
    } = req.body;
    
    // Validate required fields
    if (!bill_no || !bill_date || !account_type_bank_name) {
      return res.status(400).json({ error: 'Bill number, bill date, and account type are required' });
    }
    
    // Driver name is optional - set default value if null to avoid database constraint
    const finalDriverName = driver_name || 'N/A';
    const finalVehicleNo = vehicle_no || 'N/A';
    
    // Validate driver_id if provided
    let safeDriverId = null;
    if (driver_id && driver_id !== null && driver_id !== '') {
      try {
        const [driverCheck] = await db.query('SELECT id, name FROM drivers WHERE id = ?', [driver_id]);
        if (driverCheck.length > 0) {
          safeDriverId = driver_id;
        }
      } catch (error) {
        console.log('⚠️ Error validating driver, setting to NULL');
      }
    }
    
    // Validate bunk_id if provided
    let safeBunkId = null;
    if (bunk_id && bunk_id !== null && bunk_id !== '') {
      try {
        const [bunkCheck] = await db.query('SELECT id, bunk_name FROM bunks WHERE id = ?', [bunk_id]);
        if (bunkCheck.length > 0) {
          safeBunkId = bunk_id;
        }
      } catch (error) {
        console.log('⚠️ Error validating bunk, setting to NULL');
      }
    }
    
    // Validate bank_id if provided
    let safeBankId = null;
    if (bank_id && bank_id !== null && bank_id !== '') {
      try {
        const [bankCheck] = await db.query('SELECT id, bank_name FROM banks WHERE id = ?', [bank_id]);
        if (bankCheck.length > 0) {
          safeBankId = bank_id;
        }
      } catch (error) {
        console.log('⚠️ Error validating bank, setting to NULL');
      }
    }
    
    // Handle company_id - if it's invalid or doesn't exist, set to NULL (optional field)
    let safeCompanyId = null;
    if (company_id && company_id !== null && company_id !== '') {
      try {
        // Check if company exists in company_details table (the referenced table)
        const [companyCheck] = await db.query('SELECT id FROM company_details WHERE id = ?', [company_id]);
        if (companyCheck.length > 0) {
          safeCompanyId = company_id;
        } else {
          console.log(`Warning: Company ID ${company_id} not found in company_details table, setting to NULL`);
        }
      } catch (err) {
        console.log(`Warning: Could not verify company ID ${company_id}, setting to NULL`);
      }
    }
    
    // Check if customer_id column exists
    const [tableInfo] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'payment_entries' 
      AND COLUMN_NAME = 'customer_id'
    `);
    
    let updateQuery, updateParams;
      
      // Handle customer_id validation (optional field)
      let safeCustomerId = null;
      if (customer_id && customer_id !== null && customer_id !== '') {
        try {
          const [customerCheck] = await db.query('SELECT id FROM customers WHERE id = ?', [customer_id]);
          if (customerCheck.length > 0) {
            safeCustomerId = customer_id;
          } else {
            console.log(`Warning: Customer ID ${customer_id} not found in customers table, setting to NULL`);
          }
        } catch (err) {
          console.log(`Warning: Could not verify customer ID ${customer_id}, setting to NULL`);
        }
      }
    
    // Validate selected expense type if provided
    if (selected_expense_id) {
      const [expenseTypeCheck] = await db.query('SELECT id, name FROM expense_types WHERE id = ?', [selected_expense_id]);
      if (expenseTypeCheck.length === 0) {
        return res.status(400).json({ error: 'Selected expense type not found', expense_id: selected_expense_id });
      }
    }
    
    if (tableInfo.length > 0) {
      // customer_id column exists, include it in the update WITH new expense fields
      updateQuery = `
        UPDATE payment_entries SET 
          bill_no = ?, bill_date = ?, account_type_bank_name = ?, bank_id = ?, driver_id = ?, 
          driver_name = ?, vehicle_no = ?, bunk_id = ?, company_id = ?, customer_id = ?,
          selected_expense_id = ?, expense_amount = ?, total = ?, note = ?
        WHERE id = ?
      `;
      
      updateParams = [
        bill_no, bill_date, account_type_bank_name, safeBankId, safeDriverId, 
        finalDriverName, finalVehicleNo, safeBunkId, safeCompanyId, safeCustomerId,
        selected_expense_id || null, parseFloat(expense_amount || 0), 
        parseFloat(total || 0), note && note.trim() !== '' ? note : 'N/A',
        req.params.id
      ];
    } else {
      // customer_id column doesn't exist, use original update WITH new expense fields
      updateQuery = `
        UPDATE payment_entries SET 
          bill_no = ?, bill_date = ?, account_type_bank_name = ?, bank_id = ?, driver_id = ?, 
          driver_name = ?, vehicle_no = ?, bunk_id = ?, company_id = ?,
          selected_expense_id = ?, expense_amount = ?, total = ?, note = ?
        WHERE id = ?
      `;
      
      updateParams = [
        bill_no, bill_date, account_type_bank_name, safeBankId, safeDriverId, 
        finalDriverName, finalVehicleNo, safeBunkId, safeCompanyId,
        selected_expense_id || null, parseFloat(expense_amount || 0), 
        parseFloat(total || 0), note && note.trim() !== '' ? note : 'N/A',
        req.params.id
      ];
    }
    
    await db.query(updateQuery, updateParams);
    
    res.json({ message: 'Payment entry updated successfully' });
  } catch (err) {
    console.error('Error updating payment entry:', err);
    res.status(500).json({ error: 'Failed to update payment entry' });
  }
});

// Delete payment entry
router.delete('/:id', async (req, res) => {
  try {
    const [result] = await db.query('DELETE FROM payment_entries WHERE id = ?', [req.params.id]);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ error: 'Payment entry not found' });
    }
    
    res.json({ message: 'Payment entry deleted successfully' });
  } catch (err) {
    console.error('Error deleting payment entry:', err);
    res.status(500).json({ error: 'Failed to delete payment entry' });
  }
});

// Get all payment entries for a given vehicle_no
router.get('/by-vehicle/:vehicleNo', async (req, res) => {
  const { vehicleNo } = req.params;
  try {
    console.log('Fetching payment entries for vehicle:', vehicleNo);
    const [rows] = await db.query(`
      SELECT 
        pe.*, 
        b.bunk_name,
        DATE_FORMAT(pe.bill_date, '%Y-%m-%d') as bill_date_formatted
      FROM payment_entries pe
      LEFT JOIN bunks b ON pe.bunk_id = b.id
      WHERE LOWER(TRIM(pe.vehicle_no)) = LOWER(TRIM(?))
      ORDER BY pe.bill_date ASC
    `, [vehicleNo]);
    console.log('Found payment entries:', rows.length);
    res.json(rows);
  } catch (err) {
    console.error('Error fetching payment entries by vehicle:', err);
    res.status(500).json({ error: 'Failed to fetch payment entries by vehicle: ' + err.message });
  }
});

// Get all payment entries for a given driver_name
router.get('/by-driver/:driverName', async (req, res) => {
  const { driverName } = req.params;
  try {
    console.log('Fetching payment entries for driver:', driverName);
    const [rows] = await db.query(`
      SELECT 
        pe.*, 
        b.bunk_name,
        c.name as company_name,
        cust.name as customer_name,
        DATE_FORMAT(pe.bill_date, '%Y-%m-%d') as bill_date_formatted
      FROM payment_entries pe
      LEFT JOIN bunks b ON pe.bunk_id = b.id
      LEFT JOIN companies c ON pe.company_id = c.id
      LEFT JOIN customers cust ON pe.customer_id = cust.id
      WHERE LOWER(TRIM(pe.driver_name)) = LOWER(TRIM(?))
      ORDER BY pe.bill_date ASC
    `, [driverName]);
    console.log('Found payment entries for driver:', rows.length);
    res.json(rows);
  } catch (err) {
    console.error('Error fetching payment entries by driver:', err);
    res.status(500).json({ error: 'Failed to fetch payment entries by driver: ' + err.message });
  }
});

// Get all payment entries for a given driver_id
router.get('/by-driver-id/:driverId', async (req, res) => {
  const { driverId } = req.params;
  try {
    console.log('Fetching payment entries for driver ID:', driverId);
    const [rows] = await db.query(`
      SELECT 
        pe.*, 
        b.bunk_name,
        c.name as company_name,
        cust.name as customer_name,
        DATE_FORMAT(pe.bill_date, '%d/%m/%Y') as bill_date_formatted
      FROM payment_entries pe
      LEFT JOIN bunks b ON pe.bunk_id = b.id
      LEFT JOIN companies c ON pe.company_id = c.id
      LEFT JOIN customers cust ON pe.customer_id = cust.id
      WHERE pe.driver_id = ?
      ORDER BY pe.bill_date ASC
    `, [driverId]);
    
    console.log(`Found payment entries for driver ID ${driverId}:`, rows.length);
    res.json(rows);
  } catch (err) {
    console.error('Error fetching payment entries by driver ID:', err);
    res.status(500).json({ error: 'Failed to fetch payment entries by driver ID: ' + err.message });
  }
});

// Get all payment entries for a given company_id
router.get('/by-company/:companyId', async (req, res) => {
  const { companyId } = req.params;
  try {
    console.log('Fetching payment entries for company ID:', companyId);
    const [rows] = await db.query(`
      SELECT 
        pe.*, 
        b.bunk_name,
        DATE_FORMAT(pe.bill_date, '%Y-%m-%d') as formatted_bill_date,
        pe.amount as payment_amount
      FROM payment_entries pe
      LEFT JOIN bunks b ON pe.bunk_id = b.id
      WHERE pe.company_id = ?
      ORDER BY pe.bill_date ASC
    `, [companyId]);
    
    console.log(`Found payment entries for company ID ${companyId}:`, rows.length);
    
    // Format dates consistently
    const formattedRows = rows.map(row => ({
      ...row,
      bill_date: row.formatted_bill_date || row.bill_date,
      created_at: row.created_at ? new Date(row.created_at).toISOString().split('T')[0] : null
    }));
    
    res.json(formattedRows);
  } catch (err) {
    console.error('Error fetching payment entries by company:', err);
    res.status(500).json({ error: 'Failed to fetch payment entries by company: ' + err.message });
  }
});

// Get all payment entries for a given customer_id
// This will fetch payment entries that might be related to the customer through invoices or other relationships
router.get('/by-customer/:customerId', async (req, res) => {
  const { customerId } = req.params;
  try {
    // First, get the customer details to understand the relationship
    const [customerRows] = await db.query('SELECT * FROM customers WHERE id = ?', [customerId]);
    if (customerRows.length === 0) {
      return res.status(404).json({ error: 'Customer not found' });
    }
    
    const customer = customerRows[0];
    
    // Check if customer_id column exists in payment_entries table
    const [tableInfo] = await db.query(`
      SELECT COLUMN_NAME 
      FROM INFORMATION_SCHEMA.COLUMNS 
      WHERE TABLE_SCHEMA = DATABASE() 
      AND TABLE_NAME = 'payment_entries' 
      AND COLUMN_NAME = 'customer_id'
    `);
    
    let rows = [];
    
    if (tableInfo.length > 0) {
      // customer_id column exists, use direct relationship
      console.log('Using direct customer_id relationship for payment entries');
      [rows] = await db.query(`
        SELECT 
          pe.*, 
          b.bunk_name
        FROM payment_entries pe
        LEFT JOIN bunks b ON pe.bunk_id = b.id
        WHERE pe.customer_id = ?
        ORDER BY pe.bill_date ASC
      `, [customerId]);
    } else {
      // customer_id column doesn't exist, use relationship-based approach
      console.log('Using relationship-based approach for payment entries');
      
      // Get payment entries that might be related to this customer
      // This could include:
      // 1. Payment entries for invoices related to this customer
      // 2. Payment entries for companies that this customer works with
      // 3. Payment entries that mention this customer in notes or other fields
      
      [rows] = await db.query(`
        SELECT DISTINCT 
          pe.*, 
          b.bunk_name, 
          bk.bank_name, 
          c.name as company_name,
          CASE 
            WHEN bk.bank_name IS NOT NULL THEN bk.bank_name
            WHEN pe.account_type_bank_name IS NOT NULL THEN pe.account_type_bank_name
            ELSE 'N/A'
          END as display_bank_name
        FROM payment_entries pe
        LEFT JOIN bunks b ON pe.bunk_id = b.id
        LEFT JOIN banks bk ON pe.bank_id = bk.id
        LEFT JOIN companies c ON pe.company_id = c.id
        WHERE (
          pe.company_id IN (
            SELECT DISTINCT company_id 
            FROM invoices 
            WHERE customer_id = ? AND company_id IS NOT NULL
          )
          OR pe.driver_name LIKE ?
        )
        ORDER BY pe.bill_date ASC
      `, [customerId, `%${customer.name}%`]);
    }
    
    console.log(`Found payment entries for customer ID ${customerId}:`, rows.length);
    res.json(rows);
  } catch (err) {
    console.error('Error fetching payment entries by customer:', err);
    res.status(500).json({ error: 'Failed to fetch payment entries by customer: ' + err.message });
  }
});

module.exports = { router, setDb }; 