const express = require('express');
const router = express.Router();
let db;

const setDb = (database) => {
  db = database;
};

// GET all settings companies
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT * FROM settings_companies 
      ORDER BY created_at ASC
    `);
    res.json(rows);
  } catch (err) {
    console.error('Error fetching settings companies:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// GET single settings company by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    const [rows] = await db.query('SELECT * FROM settings_companies WHERE id = ?', [id]);
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching settings company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// CREATE new settings company
router.post('/', async (req, res) => {
  const {
    company_name,
    gst,
    phone,
    address,
    city,
    email,
    pan,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name
  } = req.body;

  if (!company_name) {
    return res.status(400).json({ message: 'Company name is required' });
  }

  try {
    const [result] = await db.query(`
      INSERT INTO settings_companies 
      (company_name, gst, phone, address, city, email, pan, bank_name, account_number, ifsc, branch, account_name)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [company_name, gst, phone, address, city, email, pan, bank_name, account_number, ifsc, branch, account_name]);

    res.status(201).json({
      message: 'Company created successfully',
      id: result.insertId
    });
  } catch (err) {
    console.error('Error creating settings company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// UPDATE settings company
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const {
    company_name,
    gst,
    phone,
    address,
    city,
    email,
    pan,
    bank_name,
    account_number,
    ifsc,
    branch,
    account_name
  } = req.body;

  if (!company_name) {
    return res.status(400).json({ message: 'Company name is required' });
  }

  try {
    const [result] = await db.query(`
      UPDATE settings_companies 
      SET company_name = ?, gst = ?, phone = ?, address = ?, city = ?, email = ?, 
          pan = ?, bank_name = ?, account_number = ?, ifsc = ?, branch = ?, account_name = ?
      WHERE id = ?
    `, [company_name, gst, phone, address, city, email, pan, bank_name, account_number, ifsc, branch, account_name, id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }

    res.json({ message: 'Company updated successfully' });
  } catch (err) {
    console.error('Error updating settings company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// DELETE settings company
router.delete('/:id', async (req, res) => {
  const { id } = req.params;
  try {
    const [result] = await db.query('DELETE FROM settings_companies WHERE id = ?', [id]);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Company not found' });
    }

    res.json({ message: 'Company deleted successfully' });
  } catch (err) {
    console.error('Error deleting settings company:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb };














