const express = require('express');
const router = express.Router();
const mysql = require('mysql2/promise');

// MySQL connection pool (will be passed from server.js)
let db;
function setDb(pool) { db = pool; }

// Get all vehicles
router.get('/', async (req, res) => {
  try {
    const [rows] = await db.query('SELECT * FROM vehicles ORDER BY created_at ASC');
    res.json(rows);
  } catch (err) {
    console.error('Error fetching vehicles:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Get single vehicle by ID
router.get('/:id', async (req, res) => {
  const { id } = req.params;
  
  try {
    const [rows] = await db.query('SELECT * FROM vehicles WHERE id = ?', [id]);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Vehicle not found' });
    }
    
    res.json(rows[0]);
  } catch (err) {
    console.error('Error fetching vehicle:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Add new vehicle
router.post('/', async (req, res) => {
  const { number, name, year } = req.body;
  
  if (!number || !name || !year) {
    return res.status(400).json({ message: 'Vehicle number, name, and year are required' });
  }

  try {
    // Check if vehicle number already exists
    const [existing] = await db.query('SELECT id FROM vehicles WHERE number = ?', [number]);
    if (existing.length > 0) {
      return res.status(400).json({ message: 'Vehicle number already exists' });
    }

    const [result] = await db.query(
      'INSERT INTO vehicles (number, name, year) VALUES (?, ?, ?)',
      [number, name, year]
    );
    
    res.status(201).json({ 
      message: 'Vehicle added successfully',
      id: result.insertId,
      vehicle: { id: result.insertId, number, name, year }
    });
  } catch (err) {
    console.error('Error adding vehicle:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Update vehicle
router.put('/:id', async (req, res) => {
  const { id } = req.params;
  const { number, name, year } = req.body;
  
  if (!number || !name || !year) {
    return res.status(400).json({ message: 'Vehicle number, name, and year are required' });
  }

  try {
    // Check if vehicle exists
    const [existing] = await db.query('SELECT id FROM vehicles WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({ message: 'Vehicle not found' });
    }

    // Check if vehicle number already exists (excluding current vehicle)
    const [duplicate] = await db.query('SELECT id FROM vehicles WHERE number = ? AND id != ?', [number, id]);
    if (duplicate.length > 0) {
      return res.status(400).json({ message: 'Vehicle number already exists' });
    }

    await db.query(
      'UPDATE vehicles SET number = ?, name = ?, year = ? WHERE id = ?',
      [number, name, year, id]
    );
    
    res.json({ message: 'Vehicle updated successfully' });
  } catch (err) {
    console.error('Error updating vehicle:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

// Delete vehicle
router.delete('/:id', async (req, res) => {
  const { id } = req.params;

  try {
    const [result] = await db.query('DELETE FROM vehicles WHERE id = ?', [id]);
    
    if (result.affectedRows === 0) {
      return res.status(404).json({ message: 'Vehicle not found' });
    }
    
    res.json({ message: 'Vehicle deleted successfully' });
  } catch (err) {
    console.error('Error deleting vehicle:', err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
});

module.exports = { router, setDb }; 